"use client"

import { useForm } from "@inertiajs/react"
import { Button } from "../Components/UI/Button"
import { Card, CardContent } from "../Components/UI/Card"
import { Input } from "../Components/UI/Input"
import { Label } from "../Components/UI/Label"
import {
  ShieldAlert,
  Key,
  User,
  AlertCircle,
  ExternalLink,
  Mail
} from "lucide-react"
import { motion } from "framer-motion"

export default function InvalidLicense({ title = "License Verification Required", note = "Please verify your license to continue" }) {
  const { data, setData, post, processing, errors } = useForm({
    purchase_code: '',
    username: ''
  })

  const handleSubmit = (e) => {
    e.preventDefault()
    post('/verify-purchase')
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-red-50 via-white to-orange-50 flex items-center justify-center p-4">
      <div className="max-w-2xl w-full">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
        >
          <Card className="backdrop-blur-sm bg-white/90 shadow-2xl border-0">
            <CardContent className="p-8 md:p-12">
              {/* Icon */}
              <div className="flex justify-center mb-6">
                <div className="w-20 h-20 bg-gradient-to-br from-red-500 to-orange-600 rounded-full flex items-center justify-center animate-pulse">
                  <ShieldAlert className="w-12 h-12 text-white" />
                </div>
              </div>

              {/* Title */}
              <h1 className="text-3xl md:text-4xl font-bold text-center text-gray-900 mb-4">
                {title}
              </h1>

              {/* Note */}
              <p className="text-center text-gray-600 text-lg mb-8">
                {note}
              </p>

              {/* Error Details */}
              <div className="bg-red-50 border border-red-200 rounded-lg p-6 mb-8">
                <h3 className="font-semibold text-red-900 mb-3 flex items-center">
                  <AlertCircle className="w-5 h-5 mr-2" />
                  License Verification Failed
                </h3>
                <p className="text-red-700 text-sm mb-4">
                  Your software license could not be verified. This application requires a valid license to operate.
                </p>
                <p className="text-sm text-red-600 font-medium mb-2">
                  Common reasons for verification failure:
                </p>
                <ul className="list-disc list-inside space-y-2 text-sm text-red-600">
                  <li>Invalid or expired purchase code</li>
                  <li>License already activated on another domain</li>
                  <li>Envato username doesn't match purchase</li>
                  <li>License has been refunded or canceled</li>
                  <li>Domain change without re-verification</li>
                </ul>
              </div>

              {/* Verification Form */}
              <form onSubmit={handleSubmit} className="space-y-6">
                {/* Purchase Code */}
                <div className="space-y-2">
                  <Label htmlFor="purchase_code" className="text-gray-700">
                    Purchase Code <span className="text-red-500">*</span>
                  </Label>
                  <div className="relative">
                    <Key className="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
                    <Input
                      id="purchase_code"
                      type="text"
                      placeholder="xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx"
                      value={data.purchase_code}
                      onChange={(e) => setData('purchase_code', e.target.value)}
                      className="pl-10 h-12"
                      required
                    />
                  </div>
                  {errors.purchase_code && (
                    <p className="text-sm text-red-600">{errors.purchase_code}</p>
                  )}
                </div>

                {/* Envato Username */}
                <div className="space-y-2">
                  <Label htmlFor="username" className="text-gray-700">
                    Envato Username <span className="text-red-500">*</span>
                  </Label>
                  <div className="relative">
                    <User className="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
                    <Input
                      id="username"
                      type="text"
                      placeholder="your_envato_username"
                      value={data.username}
                      onChange={(e) => setData('username', e.target.value)}
                      className="pl-10 h-12"
                      required
                    />
                  </div>
                  {errors.username && (
                    <p className="text-sm text-red-600">{errors.username}</p>
                  )}
                </div>

                {/* Submit Button */}
                <Button
                  type="submit"
                  disabled={processing}
                  className="w-full bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white h-12 shadow-lg hover:shadow-xl transform hover:scale-[1.02] transition-all"
                >
                  {processing ? (
                    <span className="flex items-center justify-center">
                      <svg className="animate-spin -ml-1 mr-3 h-5 w-5 text-white" fill="none" viewBox="0 0 24 24">
                        <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                        <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                      </svg>
                      Verifying License...
                    </span>
                  ) : (
                    <span className="flex items-center justify-center">
                      <ShieldAlert className="w-5 h-5 mr-2" />
                      Verify License Now
                    </span>
                  )}
                </Button>
              </form>

              {/* Help Links */}
              <div className="mt-6 space-y-3">
                <a
                  href="https://help.market.envato.com/hc/en-us/articles/202822600-Where-Is-My-Purchase-Code-"
                  target="_blank"
                  rel="noopener noreferrer"
                  className="w-full inline-flex items-center justify-center px-4 py-3 bg-gray-100 text-gray-700 text-sm font-medium rounded-lg hover:bg-gray-200 transition-all"
                >
                  <ExternalLink className="w-4 h-4 mr-2" />
                  Where to Find Purchase Code?
                </a>
              </div>

              {/* Support Info */}
              <div className="mt-8 pt-6 border-t border-gray-200">
                <p className="text-center text-sm text-gray-500 flex items-center justify-center">
                  <Mail className="w-4 h-4 mr-2" />
                  Need assistance? Contact{' '}
                  <a href="mailto:support@postuno.com" className="text-blue-600 hover:text-blue-700 font-medium ml-1">
                    support@postuno.com
                  </a>
                </p>
              </div>
            </CardContent>
          </Card>

          {/* Footer */}
          <div className="text-center mt-6">
            <p className="text-sm text-gray-500">
              Postuno - Professional Social Media Management Platform
            </p>
            <p className="text-xs text-gray-400 mt-1">
              Protected by Envato Market License
            </p>
          </div>
        </motion.div>
      </div>
    </div>
  )
}
