<?php

namespace Database\Seeders\Core;

use App\Enums\StatusEnum;
use App\Models\Core\Language;
use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\File;
use App\Helpers\InstallationLogger;

class LangSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        InstallationLogger::info('LangSeeder: Starting language seeding');

        try {
            // Check if English language already exists by country code (US for English)
            $existingLanguage = Language::where('code', 'US')->first();

            if (!$existingLanguage) {
                InstallationLogger::info('LangSeeder: Creating default English language entry');

                $language = Language::firstOrCreate(
                    ['code' => 'US'], // Country code for flag display
                    [
                        'name'       => 'English',
                        'lang_code'  => 'en', // Language code for translations
                        'ltr'        => StatusEnum::true->status(),
                        'created_by' => 'System',
                        'is_default' => StatusEnum::true->status(),
                        'status'     => StatusEnum::true->status(),
                    ]
                );

                InstallationLogger::info("LangSeeder: Created English language - ID: {$language->id}, Code: {$language->code}, Lang Code: {$language->lang_code}");

                // Create language directory and files
                $this->createLanguageFiles($language->lang_code);
                InstallationLogger::info('LangSeeder: Created language files for: ' . $language->lang_code);
            } else {
                InstallationLogger::info('LangSeeder: English language already exists, skipping');
            }

        } catch (\Exception $e) {
            InstallationLogger::error('LangSeeder: Failed with error: ' . $e->getMessage());
            InstallationLogger::error('LangSeeder: File: ' . $e->getFile() . ' Line: ' . $e->getLine());
            throw $e;
        }

        InstallationLogger::info('LangSeeder: Completed successfully');
    }

    /**
     * Create language directory and comprehensive translation files
     */
    private function createLanguageFiles($langCode)
    {
        $langPath = resource_path("lang/{$langCode}");

        if (!File::exists($langPath)) {
            File::makeDirectory($langPath, 0755, true);
        }

        $filePath = $langPath . "/messages.php";

        if (!File::exists($filePath)) {
            $content = <<<'PHP'
<?php

return [
    // Common
    'welcome' => 'Welcome',
    'hello' => 'Hello',
    'dashboard' => 'Dashboard',
    'settings' => 'Settings',
    'profile' => 'Profile',
    'logout' => 'Logout',
    'login' => 'Login',
    'register' => 'Register',
    'save' => 'Save',
    'cancel' => 'Cancel',
    'delete' => 'Delete',
    'edit' => 'Edit',
    'create' => 'Create',
    'update' => 'Update',
    'search' => 'Search',
    'filter' => 'Filter',
    'all' => 'All',
    'active' => 'Active',
    'inactive' => 'Inactive',
    'status' => 'Status',
    'actions' => 'Actions',
    'name' => 'Name',
    'email' => 'Email',
    'password' => 'Password',
    'confirm_password' => 'Confirm Password',
    'username' => 'Username',
    'phone' => 'Phone',
    'address' => 'Address',
    'country' => 'Country',
    'submit' => 'Submit',
    'close' => 'Close',
    'back' => 'Back',
    'next' => 'Next',
    'previous' => 'Previous',
    'loading' => 'Loading...',
    'no_data' => 'No data available',
    
    // Settings
    'app_settings' => 'App Settings',
    'site_name' => 'Site Name',
    'time_zone' => 'Time Zone',
    'date_format' => 'Date Format',
    'language_region' => 'Language & Region',
    'email_configuration' => 'Email Configuration',
    'security_privacy' => 'Security & Privacy',
    'backup_settings' => 'Backup Settings',
    'automation_settings' => 'Automation Settings',
    
    // Email Settings
    'smtp_host' => 'SMTP Host',
    'smtp_port' => 'SMTP Port',
    'smtp_username' => 'SMTP Username',
    'smtp_password' => 'SMTP Password',
    'smtp_encryption' => 'SMTP Encryption',
    'mail_from_address' => 'Mail From Address',
    'mail_from_name' => 'Mail From Name',
    'email_notifications' => 'Email Notifications',
    'password_reset_emails' => 'Password Reset Emails',
    'account_verification_emails' => 'Account Verification Emails',
    'general_notifications' => 'General Notifications',
    'test_mode' => 'Test Mode',
    'send_password_reset_emails_to_users' => 'Send password reset emails to users',
    'send_verification_emails_for_new_accounts' => 'Send verification emails for new accounts',
    'send_general_notification_emails' => 'Send general notification emails',
    'enable_test_mode_for_email_sending' => 'Enable test mode for email sending',
    
    // Security Settings
    'authentication_security' => 'Authentication Security',
    'password_security' => 'Password Security',
    'privacy_data' => 'Privacy & Data',
    'two_factor_authentication' => 'Two-Factor Authentication',
    'enforce_2fa_for_all_users' => 'Enforce 2FA for All Users',
    'configure_2fa' => 'Configure 2FA',
    'session_timeout' => 'Session Timeout',
    'max_login_attempts' => 'Max Login Attempts',
    'lockout_duration' => 'Lockout Duration',
    'security_notifications' => 'Security Notifications',
    'password_min_length' => 'Password Min Length',
    'require_uppercase_letters' => 'Require Uppercase Letters',
    'require_numbers' => 'Require Numbers',
    'require_special_characters' => 'Require Special Characters',
    'privacy_mode' => 'Privacy Mode',
    'data_retention_days' => 'Data Retention (Days)',
    'gdpr_compliance' => 'GDPR Compliance',
    'data_export' => 'Data Export',
    'account_deletion' => 'Account Deletion',
    'audit_logs' => 'Audit Logs',
    'require_all_users_to_enable_two_factor_authentication' => 'Require all users to enable two-factor authentication',
    'add_an_extra_layer_of_security_to_your_account' => 'Add an extra layer of security to your account',
    'send_notifications_for_security_events' => 'Send notifications for security events',
    'password_must_contain_uppercase_letters' => 'Password must contain uppercase letters',
    'password_must_contain_numbers' => 'Password must contain numbers',
    'password_must_contain_special_characters' => 'Password must contain special characters',
    'hide_sensitive_information_in_logs' => 'Hide sensitive information in logs',
    'enable_gdpr_compliance_features' => 'Enable GDPR compliance features',
    'allow_users_to_export_their_data' => 'Allow users to export their data',
    'allow_users_to_delete_their_accounts' => 'Allow users to delete their accounts',
    'enable_detailed_audit_logging' => 'Enable detailed audit logging',
    
    // Languages
    'languages' => 'Languages',
    'add_language' => 'Add Language',
    'create_language' => 'Create Language',
    'select_a_language_and_text_direction_for_your_app' => 'Select a language and text direction for your app.',
    'language' => 'Language',
    'select_language' => 'Select language',
    'text_direction' => 'Text Direction',
    'select_direction' => 'Select direction',
    'lang_code' => 'Lang Code',
    'ltr_rtl' => 'LTR/RTL',
    'translate' => 'Translate',
    'make_default' => 'Make Default',
    'search_by_name_or_code' => 'Search by name or code',
    'delete_this_language' => 'Delete this Language?',
    'are_you_really_sure_you_want_to_delete_this_language' => 'Are you really sure you want to delete this language?',
    'manage_languages_and_regional_settings' => 'Manage languages and regional settings',
    
    // Messages
    'success' => 'Success',
    'error' => 'Error',
    'warning' => 'Warning',
    'info' => 'Info',
    'language_saved' => 'Language saved',
    'language_deleted' => 'Language deleted',
    'language_updated' => 'Language updated',
    'cannot_delete_the_default_language' => 'Cannot delete the default language. Please set another language as default first.',
    
    // Pagination
    'showing' => 'Showing',
    'of' => 'of',
    'results' => 'results',
    'page' => 'Page',
    'per_page' => 'Per page',
    
    // Date/Time
    'today' => 'Today',
    'yesterday' => 'Yesterday',
    'last_week' => 'Last week',
    'last_month' => 'Last month',
    'created_at' => 'Created at',
    'updated_at' => 'Updated at',
];
PHP;

            File::put($filePath, $content);
        }
    }
}
