"use client"

import { useState } from "react"
import { router, useForm } from "@inertiajs/react"
import { Button } from "../../Components/UI/Button"
import { Card, CardContent } from "../../Components/UI/Card"
import { Input } from "../../Components/UI/Input"
import { Label } from "../../Components/UI/Label"
import { 
  Key, 
  ArrowRight,
  ArrowLeft,
  Shield,
  AlertCircle,
  CheckCircle2,
  User
} from "lucide-react"
import { motion } from "framer-motion"

export default function LicenseVerification({ appName = "Postuno", errors = {}, flash = {}, old = {} }) {
  const { data, setData, post, processing } = useForm({
    purchase_code: old.purchase_code || '',
    username: old.username || ''
  })

  const handleSubmit = (e) => {
    e.preventDefault()
    post('/install/purchase-code/verification')
  }

  const handleBack = () => {
    router.visit('/install/db-setup?verify_token=' + btoa('dbsetup_'))
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-purple-50 dark:from-gray-900 dark:via-gray-800 dark:to-gray-900">
      <div className="container mx-auto px-4 py-12">
        <div className="max-w-4xl mx-auto">
          {/* Header */}
          <motion.div 
            initial={{ opacity: 0, y: -20 }}
            animate={{ opacity: 1, y: 0 }}
            className="text-center mb-12"
          >
            <div className="inline-flex items-center justify-center w-20 h-20 rounded-full bg-gradient-to-r from-blue-500 to-purple-600 mb-6">
              <Key className="w-10 h-10 text-white" />
            </div>
            <h1 className="text-4xl font-bold text-gray-900 dark:text-white mb-4">
              License Verification
            </h1>
            <p className="text-lg text-gray-600 dark:text-gray-400">
              Enter your purchase details to activate {appName}
            </p>
          </motion.div>

          {/* Progress Bar */}
          <div className="mb-8">
            <div className="flex items-center justify-between mb-2">
              <span className="text-sm text-gray-600 dark:text-gray-400">Installation Progress</span>
              <span className="text-sm text-gray-600 dark:text-gray-400">Step 3 of 6</span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-2 dark:bg-gray-700">
              <div className="bg-gradient-to-r from-blue-500 to-purple-600 h-2 rounded-full" style={{ width: '50%' }}></div>
            </div>
          </div>

          {/* Main Card */}
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.2 }}
          >
            <Card className="backdrop-blur-sm bg-white/90 dark:bg-gray-800/90 shadow-xl border-0">
              <CardContent className="p-8">
                <form onSubmit={handleSubmit} className="space-y-6">
                  {/* Info Box */}
                  <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4 mb-6">
                    <div className="flex items-start">
                      <Shield className="w-5 h-5 text-blue-500 mt-0.5 mr-3" />
                      <div>
                        <p className="text-blue-800 dark:text-blue-300 font-medium">
                          Secure Activation
                        </p>
                        <p className="text-blue-600 dark:text-blue-400 text-sm mt-1">
                          Your license information is encrypted and securely validated.
                        </p>
                      </div>
                    </div>
                  </div>

                  {/* Purchase Code Input */}
                  <div className="space-y-2">
                    <Label htmlFor="purchase_code" className="text-gray-700 dark:text-gray-300">
                      Purchase Code <span className="text-red-500">*</span>
                    </Label>
                    <div className="relative">
                      <Key className="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
                      <Input
                        id="purchase_code"
                        type="text"
                        placeholder="Enter your purchase code"
                        value={data.purchase_code}
                        onChange={(e) => setData('purchase_code', e.target.value)}
                        className="pl-10 h-12"
                        required
                      />
                    </div>
                    {errors.purchase_code && (
                      <p className="text-sm text-red-500 mt-1">{errors.purchase_code}</p>
                    )}
                    <p className="text-xs text-gray-500 dark:text-gray-400">
                      You can find your purchase code in your Envato/CodeCanyon downloads section
                    </p>
                  </div>

                  {/* Username Input */}
                  <div className="space-y-2">
                    <Label htmlFor="username" className="text-gray-700 dark:text-gray-300">
                      Envato Username <span className="text-red-500">*</span>
                    </Label>
                    <div className="relative">
                      <User className="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
                      <Input
                        id="username"
                        type="text"
                        placeholder="Enter your Envato username"
                        value={data.username}
                        onChange={(e) => setData('username', e.target.value)}
                        className="pl-10 h-12"
                        required
                      />
                    </div>
                    {errors.username && (
                      <p className="text-sm text-red-500 mt-1">{errors.username}</p>
                    )}
                    <p className="text-xs text-gray-500 dark:text-gray-400">
                      Your Envato/CodeCanyon account username
                    </p>
                  </div>

                  {/* Error Message */}
                  {(errors.general || flash.error) && (
                    <div className="p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg">
                      <div className="flex items-start">
                        <AlertCircle className="w-5 h-5 text-red-500 mt-0.5 mr-3" />
                        <div>
                          <p className="text-red-800 dark:text-red-300 font-medium">
                            Verification Failed
                          </p>
                          <p className="text-red-600 dark:text-red-400 text-sm mt-1">
                            {errors.general || flash.error}
                          </p>
                        </div>
                      </div>
                    </div>
                  )}

                  {/* Help Section */}
                  <div className="bg-gray-50 dark:bg-gray-900/50 rounded-lg p-4">
                    <h4 className="font-medium text-gray-900 dark:text-white mb-2">
                      Where to find your purchase code?
                    </h4>
                    <ol className="text-sm text-gray-600 dark:text-gray-400 space-y-1">
                      <li>1. Log in to your Envato account</li>
                      <li>2. Go to Downloads section</li>
                      <li>3. Find {appName} in your purchases</li>
                      <li>4. Click "Download" → "License certificate & purchase code"</li>
                    </ol>
                  </div>

                  {/* Action Buttons */}
                  <div className="flex gap-4 pt-4">
                    <Button
                      type="button"
                      onClick={handleBack}
                      variant="outline"
                      className="flex-1"
                      disabled={processing}
                    >
                      <ArrowLeft className="mr-2 h-4 w-4" />
                      Back
                    </Button>
                    <Button
                      type="submit"
                      disabled={processing || !data.purchase_code || !data.username}
                      className="flex-1 bg-gradient-to-r from-blue-500 to-purple-600 hover:from-blue-600 hover:to-purple-700 text-white"
                    >
                      {processing ? (
                        <span className="flex items-center justify-center">
                          <svg className="animate-spin -ml-1 mr-3 h-5 w-5 text-white" fill="none" viewBox="0 0 24 24">
                            <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                            <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                          </svg>
                          Verifying...
                        </span>
                      ) : (
                        <span className="flex items-center justify-center">
                          Verify License
                          <ArrowRight className="ml-2 h-4 w-4" />
                        </span>
                      )}
                    </Button>
                  </div>
                </form>
              </CardContent>
            </Card>
          </motion.div>
        </div>
      </div>
    </div>
  )
}