import { ThumbsUp, MessageCircle, Share, MoreHorizontal } from "lucide-react"

// Utility functions for hashtag detection and highlighting
const extractHashtags = (text) => {
  const hashtagRegex = /#[a-zA-Z0-9_]+/g
  return text.match(hashtagRegex) || []
}

const highlightContent = (text) => {
  const hashtagRegex = /#[a-zA-Z0-9_]+/g
  const urlRegex = /(https?:\/\/[^\s]+)/g

  // Split by both hashtags and URLs, keeping the delimiters
  const parts = text.split(/(#[a-zA-Z0-9_]+|https?:\/\/[^\s]+)/g)

  return parts
    .map((part, index) => {
      if (!part) return null

      if (hashtagRegex.test(part)) {
        return (
          <span
            key={index}
            className="bg-yellow-200 dark:bg-yellow-600 text-yellow-800 dark:text-yellow-200 px-1 rounded"
          >
            {part}
          </span>
        )
      }

      if (urlRegex.test(part)) {
        return (
          <span key={index} className="text-blue-600 dark:text-blue-400">
            {part}
          </span>
        )
      }

      return part
    })
    .filter(Boolean)
}

export default function ThreadsPreview({ postContent = "", postType = "post", selectedMedia = [] }) {
  // Mock user data - in real app this would come from selected account
  const mockUser = {
    name: "James Clerk",
    username: "@jamesclerk",
    avatar: "/diverse-user-avatars.png",
    verified: false,
  }

  const hashtags = extractHashtags(postContent)

  return (
    <div className="bg-white dark:bg-zinc-900 rounded-lg border border-gray-200 dark:border-zinc-800 overflow-hidden min-h-[320px] w-full flex flex-col">
      {/* Header */}
      <div className="p-4 border-b border-gray-100 dark:border-gray-700">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-3">
            <img src={mockUser.avatar || "/placeholder.svg"} alt={mockUser.name} className="w-8 h-8 rounded-full" />
            <span className="font-medium text-gray-900 dark:text-white">{mockUser.username}</span>
          </div>
          <MoreHorizontal className="w-5 h-5 text-gray-600 dark:text-gray-400" />
        </div>
      </div>

      {/* Content */}
      <div className="p-4">
        <div className="text-gray-900 dark:text-white whitespace-pre-wrap mb-4 break-words">
          {postContent ? highlightContent(postContent) : "What's in your mind"}
        </div>

        {hashtags.length > 0 && (
          <div className="flex flex-wrap gap-2 mb-4">
            {hashtags.map((hashtag, index) => (
              <span
                key={index}
                className="bg-gray-100 dark:bg-gray-700 text-gray-800 dark:text-gray-200 px-2 py-1 rounded-full text-sm font-medium"
              >
                {hashtag}
              </span>
            ))}
          </div>
        )}

        {selectedMedia.length > 0 ? (
          <div className="mb-4">
            {selectedMedia.length === 1 ? (
              <img
                src={selectedMedia[0].thumbnail || selectedMedia[0].url}
                alt={selectedMedia[0].alt || "Post image"}
                className="w-full rounded-lg object-cover max-h-96"
              />
            ) : (
              <div className="grid grid-cols-2 gap-1 rounded-lg overflow-hidden">
                {selectedMedia.slice(0, 4).map((media, index) => (
                  <div key={media.id} className="relative aspect-square">
                    <img
                      src={media.thumbnail || media.url}
                      alt={media.alt || `Post image ${index + 1}`}
                      className="w-full h-full object-cover"
                    />
                    {index === 3 && selectedMedia.length > 4 && (
                      <div className="absolute inset-0 bg-black/60 flex items-center justify-center">
                        <span className="text-white text-xl font-bold">+{selectedMedia.length - 4}</span>
                      </div>
                    )}
                  </div>
                ))}
              </div>
            )}
          </div>
        ) : null}
      </div>

      {/* Actions */}
      <div className="border-t border-gray-100 dark:border-gray-700 p-4 mt-auto">
        <div className="flex justify-around items-center">
          <button className="flex items-center gap-2 px-4 py-2 text-gray-600 dark:text-gray-400 hover:bg-gray-50 dark:hover:bg-gray-700 rounded-lg transition-colors flex-1 justify-center">
            <ThumbsUp className="w-4 h-4" />
            <span className="text-sm">Like</span>
          </button>
          <button className="flex items-center gap-2 px-4 py-2 text-gray-600 dark:text-gray-400 hover:bg-gray-50 dark:hover:bg-gray-700 rounded-lg transition-colors flex-1 justify-center">
            <MessageCircle className="w-4 h-4" />
            <span className="text-sm">Comment</span>
          </button>
          <button className="flex items-center gap-2 px-4 py-2 text-gray-600 dark:text-gray-400 hover:bg-gray-50 dark:hover:bg-gray-700 rounded-lg transition-colors flex-1 justify-center">
            <Share className="w-4 h-4" />
            <span className="text-sm">Share</span>
          </button>
        </div>
      </div>
    </div>
  )
}
