<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Platform;
use Database\Seeders\PlatformSeeder;

class TestPlatformSeeder extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'test:platform-seeder {--reset : Reset platforms before testing}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test platform seeder functionality';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Testing Platform Seeder...');
        $this->newLine();

        // Reset platforms if requested
        if ($this->option('reset')) {
            $this->info('🔄 Resetting platforms table...');
            Platform::truncate();
            $this->info('✅ Platforms table cleared');
            $this->newLine();
        }

        // Show current state
        $this->info('=== Current State ===');
        $currentCount = Platform::count();
        $this->info("Current platforms in database: {$currentCount}");

        if ($currentCount > 0) {
            $existingPlatforms = Platform::pluck('name', 'slug')->toArray();
            foreach ($existingPlatforms as $slug => $name) {
                $this->line("  - {$name} (slug: {$slug})");
            }
        }

        $this->newLine();

        // Run the seeder
        $this->info('🚀 Running PlatformSeeder...');
        try {
            $seeder = new PlatformSeeder();
            $seeder->run();
            $this->info('✅ PlatformSeeder completed');
        } catch (\Exception $e) {
            $this->error('❌ PlatformSeeder failed: ' . $e->getMessage());
            $this->error('File: ' . $e->getFile() . ' Line: ' . $e->getLine());
            return 1;
        }

        $this->newLine();

        // Show final state
        $this->info('=== Final State ===');
        $finalCount = Platform::count();
        $this->info("Final platforms in database: {$finalCount}");

        $allPlatforms = Platform::select('id', 'name', 'slug', 'status', 'is_feature', 'is_integrated')->get();

        foreach ($allPlatforms as $platform) {
            $status = $platform->status === '1' ? 'Active' : 'Inactive';
            $feature = $platform->is_feature === '1' ? 'Featured' : 'Standard';
            $integrated = $platform->is_integrated === '1' ? 'Integrated' : 'Not Integrated';

            $this->line("  📱 {$platform->name}");
            $this->line("     ID: {$platform->id} | Slug: {$platform->slug}");
            $this->line("     Status: {$status} | Feature: {$feature} | Integration: {$integrated}");
            $this->newLine();
        }

        // Summary
        $expectedCount = 7; // Facebook, Instagram, Twitter, LinkedIn, TikTok, YouTube, Threads
        if ($finalCount === $expectedCount) {
            $this->info("✅ SUCCESS: All {$expectedCount} platforms were seeded correctly!");
        } else {
            $this->error("❌ ISSUE: Expected {$expectedCount} platforms, but found {$finalCount}");
            return 1;
        }

        return 0;
    }
}