"use client"
import { usePage } from "@inertiajs/react"
import { useState, useEffect } from "react"
import { router } from "@inertiajs/react"
import { route } from "ziggy-js"
import { Input } from "@/Components/UI/Input"
import { Button } from "@/Components/UI/Button"
import { Checkbox } from "@/Components/UI/Checkbox"
import { Eye, EyeOff, Info } from "lucide-react"
import AuthLayout from "@/Layouts/AuthLayout"
import useTranslate from "@/hooks/useTranslate";

export default function Login() {
  const { errors, site_settings, app_mode, demo_credentials } = usePage().props
  const siteName = site_settings?.site_name || 'PostUno'
  const isDemo = app_mode === 'demo'

  const [form, setForm] = useState({
    loginIdentifier: "",
    password: "",
    remember: false,
  })
  const [showPassword, setShowPassword] = useState(false)
  const [isSubmitting, setIsSubmitting] = useState(false)

  // Pre-fill demo credentials when in demo mode
  useEffect(() => {
    if (isDemo && demo_credentials) {
      setForm({
        loginIdentifier: demo_credentials.username || "",
        password: demo_credentials.password || "",
        remember: false,
      })
    }
  }, [isDemo, demo_credentials])

  const handleChange = (e) => {
    const { name, type, checked, value } = e.target
    setForm({
      ...form,
      [name]: type === "checkbox" ? checked : value,
    })
  }

  const t = useTranslate()

  const handleSubmit = (e) => {
    e.preventDefault()
    setIsSubmitting(true)

    const url = route("auth.authenticate")

    router.post(url, form ,  {
      onFinish: () => setIsSubmitting(false),
      onError: () => setIsSubmitting(false),
    })
  }

  return (
    <AuthLayout>
      {" "}
      {/* Wrap the content with AuthLayout */}

      {/* Demo Mode Banner */}
      {isDemo && (
        <div className="mb-4 lg:mb-6 p-3 lg:p-4 bg-gradient-to-r from-yellow-50 to-amber-50 dark:from-yellow-900/20 dark:to-amber-900/20 border border-yellow-200 dark:border-yellow-800 rounded-xl">
          <div className="flex items-start space-x-3">
            <Info className="w-4 h-4 lg:w-5 lg:h-5 text-yellow-600 dark:text-yellow-400 mt-0.5 flex-shrink-0" />
            <div className="min-w-0 flex-1">
              <h3 className="text-xs lg:text-sm font-semibold text-yellow-800 dark:text-yellow-200 mb-1">
                {t("Demo Mode")}
              </h3>
              <p className="text-xs lg:text-sm text-yellow-700 dark:text-yellow-300 mb-2 leading-relaxed">
                {t("You're viewing PostUno in demo mode. Login credentials have been pre-filled for easy access.")}
              </p>
              <div className="text-xs text-yellow-600 dark:text-yellow-400 space-y-1">
                <div className="truncate"><strong>{t("Email")}:</strong> {demo_credentials?.username}</div>
                <div><strong>{t("Password")}:</strong> {demo_credentials?.password}</div>
              </div>
            </div>
          </div>
        </div>
      )}
      <div className="text-center mb-4 lg:mb-6 xl:mb-8">
        <div className="mb-3 lg:mb-4">
          <div className="w-14 h-14 lg:w-16 lg:h-16 xl:w-18 xl:h-18 2xl:w-20 2xl:h-20 bg-gradient-to-br from-primary via-secondary to-primary rounded-2xl flex items-center justify-center mx-auto mb-3 lg:mb-4 shadow-xl shadow-primary/30 dark:shadow-primary/20 transform hover:scale-105 transition-all duration-300">
            <div className="w-7 h-7 lg:w-8 lg:h-8 xl:w-9 xl:h-9 2xl:w-10 2xl:h-10 bg-white/90 backdrop-blur-sm rounded-xl flex items-center justify-center shadow-inner">
              <div className="w-3.5 h-3.5 lg:w-4 lg:h-4 xl:w-4.5 xl:h-4.5 2xl:w-5 2xl:h-5 bg-gradient-to-br from-primary to-secondary rounded-sm animate-pulse"></div>
            </div>
          </div>
        </div>
        <h1 className="text-xl lg:text-2xl xl:text-3xl 2xl:text-4xl font-bold mb-2 lg:mb-3 bg-gradient-to-r from-primary via-secondary to-primary bg-clip-text text-transparent leading-tight">
          {t("Welcome to")} {siteName}
        </h1>
        <p className="text-xs lg:text-sm xl:text-base 2xl:text-lg text-gray-500 dark:text-gray-400 font-medium tracking-wide">
          {t("Your social media command center")}
        </p>
        <div className="w-16 lg:w-20 xl:w-24 2xl:w-28 h-1 bg-gradient-to-r from-primary to-secondary rounded-full mx-auto mt-3 lg:mt-4 opacity-60"></div>
      </div>
      <form onSubmit={handleSubmit} noValidate className="space-y-3 lg:space-y-4 xl:space-y-5">
        <div className="group">
          <label htmlFor="loginIdentifier" className="block mb-2 lg:mb-3 text-sm lg:text-base font-semibold text-gray-700 dark:text-gray-200 transition-colors duration-200 group-focus-within:text-primary">
            {t("Email")}
          </label>
          <div className="relative">
            <Input
              type="email"
              id="loginIdentifier"
              name="loginIdentifier"
              value={form.loginIdentifier}
              onChange={handleChange}
              placeholder={isDemo ? demo_credentials?.username : t("Enter your email address")}
              className={`w-full px-4 py-3 lg:px-5 lg:py-4 xl:px-6 xl:py-5 border rounded-xl focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:text-white text-base lg:text-lg backdrop-blur-sm transition-all duration-300 placeholder:text-gray-400 dark:placeholder:text-gray-500 ${
                isDemo
                  ? 'bg-green-50/80 dark:bg-green-900/20 border-green-200 dark:border-green-800 hover:bg-green-50 dark:hover:bg-green-900/30 hover:border-green-300'
                  : 'bg-white/50 dark:bg-zinc-800/50 border-gray-200 dark:border-zinc-700 hover:bg-white/70 dark:hover:bg-zinc-800/70 hover:border-primary/50'
              }`}
              readOnly={isDemo}
            />
            <div className="absolute inset-0 rounded-xl bg-gradient-to-r from-primary/10 to-secondary/10 opacity-0 group-focus-within:opacity-100 transition-opacity duration-300 pointer-events-none"></div>
          </div>
          {errors.loginIdentifier && <p className="mt-2 text-red-500 text-sm lg:text-base font-medium">{errors.loginIdentifier}</p>}
        </div>
        <div className="group">
          <label htmlFor="password" className="block mb-2 lg:mb-3 text-sm lg:text-base font-semibold text-gray-700 dark:text-gray-200 transition-colors duration-200 group-focus-within:text-primary">
            {t("Password")}
          </label>
          <div className="relative">
            <Input
              type={showPassword ? "text" : "password"}
              id="password"
              name="password"
              value={form.password}
              onChange={handleChange}
              placeholder={isDemo ? demo_credentials?.password : t("Enter your password")}
              className={`w-full px-4 py-3 pr-12 lg:px-5 lg:py-4 lg:pr-14 xl:px-6 xl:py-5 xl:pr-16 border rounded-xl focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:text-white text-base lg:text-lg backdrop-blur-sm transition-all duration-300 placeholder:text-gray-400 dark:placeholder:text-gray-500 ${
                isDemo
                  ? 'bg-green-50/80 dark:bg-green-900/20 border-green-200 dark:border-green-800 hover:bg-green-50 dark:hover:bg-green-900/30 hover:border-green-300'
                  : 'bg-white/50 dark:bg-zinc-800/50 border-gray-200 dark:border-zinc-700 hover:bg-white/70 dark:hover:bg-zinc-800/70 hover:border-primary/50'
              }`}
              readOnly={isDemo}
            />
            <button
              type="button"
              onClick={() => setShowPassword(!showPassword)}
              className="absolute inset-y-0 right-0 pr-4 lg:pr-5 xl:pr-6 flex items-center text-gray-400 hover:text-primary dark:text-gray-500 dark:hover:text-primary transition-all duration-200 hover:scale-110"
              aria-label={showPassword ? "Hide password" : "Show password"}
            >
              {showPassword ? <EyeOff className="h-5 w-5 lg:h-6 lg:w-6" /> : <Eye className="h-5 w-5 lg:h-6 lg:w-6" />}
            </button>
            <div className="absolute inset-0 rounded-xl bg-gradient-to-r from-primary/10 to-secondary/10 opacity-0 group-focus-within:opacity-100 transition-opacity duration-300 pointer-events-none"></div>
          </div>
          {errors.password && <p className="mt-2 text-red-500 text-sm lg:text-base font-medium">{errors.password}</p>}
        </div>
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-2 sm:gap-0">
          <div className="flex items-center">
            <Checkbox
              id="remember"
              name="remember"
              checked={form.remember}
              onCheckedChange={(checked) => setForm({ ...form, remember: checked })}
              className="mr-2"
            />
            <label htmlFor="remember" className="text-xs sm:text-sm text-gray-700 dark:text-gray-200">
              {t("Remember me")}
            </label>
          </div>
          <a href={route('auth.password.request')} className="text-xs sm:text-sm text-primary hover:text-secondary hover:underline transition-colors">
            {t("Forgot Password?")}
          </a>
        </div>
        <Button
          type="submit"
          isLoading={isSubmitting}
          className="w-full py-4 lg:py-5 xl:py-6 rounded-xl font-bold text-white bg-gradient-to-r from-primary via-secondary to-primary hover:from-secondary hover:via-primary hover:to-secondary focus:outline-none focus:ring-4 focus:ring-primary/30 transition-all duration-300 shadow-xl hover:shadow-2xl hover:shadow-primary/25 transform hover:scale-[1.02] active:scale-[0.98] backdrop-blur-sm text-base lg:text-lg xl:text-xl"
        >
          {isSubmitting ? t("Signing in...") : t("Sign In")}
        </Button>
      </form>
    </AuthLayout>
  )
}