"use client"

import { motion } from "framer-motion"
import { cn } from "@/Lib/utils"

export default function PageHeader({ 
  icon: Icon, 
  title, 
  description, 
  actions, 
  stats,
  className,
  variant = "default" 
}) {
  const variants = {
    default: "bg-white dark:bg-zinc-900 border-gray-200 dark:border-zinc-800",
    gradient: "bg-gradient-to-br from-white to-gray-50 dark:from-gray-900 dark:to-gray-800 border-gray-200 dark:border-zinc-800",
    minimal: "bg-transparent border-transparent"
  }

  return (
    <motion.div 
      initial={{ opacity: 0, y: -10 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ duration: 0.4, ease: "easeOut" }}
      className={cn(
        "rounded-xl p-6 mb-6 border shadow-sm",
        variants[variant],
        className
      )}
    >
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-4">
          {Icon && (
            <div>
              <Icon className="w-8 h-8 text-primary" />
            </div>
          )}
          <div className="space-y-1">
            <h1 className="lg:text-3xl text-2xl font-bold bg-gradient-to-r from-gray-900 to-gray-700 dark:from-white dark:to-gray-300 bg-clip-text text-transparent">
              {title}
            </h1>
            {description && (
              <p className="text-gray-600 dark:text-gray-400 text-lg">
                {description}
              </p>
            )}
          </div>
        </div>
        
        <div className="flex items-center gap-6">
          {/* Stats Section */}
          {stats && stats.length > 0 && (
            <>
              {stats.map((stat, index) => (
                <div key={index} className="text-right">
                  <p className="text-sm text-gray-500 dark:text-gray-400">{stat.label}</p>
                  <p className="text-2xl font-bold text-gray-900 dark:text-white">{stat.value}</p>
                </div>
              ))}
              {actions && <div className="h-12 w-px bg-gray-200 dark:bg-gray-700"></div>}
            </>
          )}
          
          {/* Actions Section */}
          {actions && (
            <div className="flex items-center gap-3">
              {actions}
            </div>
          )}
        </div>
      </div>
    </motion.div>
  )
}

// Sub-components for better organization
export function PageHeaderAction({ children, variant = "primary", className, ...props }) {
  const variants = {
    primary: "bg-gradient-to-r from-primary to-secondary text-white hover:from-secondary hover:to-primary shadow-lg hover:shadow-xl transform hover:-translate-y-0.5 transition-all duration-200",
    secondary: "bg-white dark:bg-zinc-900 text-gray-700 dark:text-gray-200 border border-gray-300 dark:border-gray-600 hover:bg-gray-50 dark:hover:bg-gray-700",
    danger: "bg-red-500 text-white hover:bg-red-600"
  }

  return (
    <button
      className={cn(
        "inline-flex items-center gap-2 px-4 py-2 rounded-lg font-medium text-sm transition-all duration-200",
        variants[variant],
        className
      )}
      {...props}
    >
      {children}
    </button>
  )
}

export function PageHeaderStat({ label, value, icon: Icon, trend }) {
  return (
    <div className="text-right space-y-1">
      <div className="flex items-center justify-end gap-2">
        {Icon && <Icon className="w-4 h-4 text-gray-500" />}
        <p className="text-sm text-gray-500 dark:text-gray-400">{label}</p>
      </div>
      <div className="flex items-center justify-end gap-2">
        <p className="text-2xl font-bold text-gray-900 dark:text-white">{value}</p>
        {trend && (
          <span className={cn(
            "text-xs px-2 py-1 rounded-full font-medium",
            trend > 0 ? "bg-green-100 text-green-700" : "bg-red-100 text-red-700"
          )}>
            {trend > 0 ? "+" : ""}{trend}%
          </span>
        )}
      </div>
    </div>
  )
}