"use client"

import { useState } from "react"

const FileUpload = ({
    label,
    value,
    onChange,
    accept = "image/*",
    placeholder = "No file chosen",
    helperText,
    showViewLink = false,
    onViewClick,
    className = "",
}) => {
    const [dragOver, setDragOver] = useState(false)

    const handleFileChange = (e) => {
        const file = e.target.files[0]
        if (onChange) {
            onChange(file)
        }
    }

    const handleDragOver = (e) => {
        e.preventDefault()
        setDragOver(true)
    }

    const handleDragLeave = (e) => {
        e.preventDefault()
        setDragOver(false)
    }

    const handleDrop = (e) => {
        e.preventDefault()
        setDragOver(false)
        const file = e.dataTransfer.files[0]
        if (file && onChange) {
            onChange(file)
        }
    }

    return (
        <div className={className}>
            {label && (
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-200 mb-2">
                    {label}
                </label>
            )}
            <div className="space-y-2">
                <div className="relative">
                    <input
                        type="file"
                        accept={accept}
                        onChange={handleFileChange}
                        className="absolute inset-0 w-full h-full opacity-0 cursor-pointer z-10"
                    />
                    <div
                        className={`flex border rounded-md overflow-hidden transition-colors
                        ${dragOver ? "border-primary bg-primary/10 dark:bg-primary/20" : "border-gray-200 bg-gray-50 dark:border-gray-700 dark:bg-gray-800"}`}
                        onDragOver={handleDragOver}
                        onDragLeave={handleDragLeave}
                        onDrop={handleDrop}
                    >
                        <div className="bg-gray-200 dark:bg-gray-700 px-3 py-2 text-sm text-gray-600 dark:text-gray-300 border-r border-gray-300 dark:border-gray-600">
                            Choose file
                        </div>
                        <div className="flex-1 px-3 py-2 text-sm text-gray-500 dark:text-gray-400">
                            {value ? value.name : placeholder}
                        </div>
                    </div>
                </div>
                {helperText && (
                    <p className="text-xs text-gray-500 dark:text-gray-400">
                        {helperText}
                        {showViewLink && onViewClick && (
                            <>
                                {" "}
                                <span
                                    className="text-primary cursor-pointer hover:text-primary-dark"
                                    onClick={onViewClick}
                                >
                                    View image
                                </span>
                            </>
                        )}
                    </p>
                )}
            </div>
        </div>
    )
}

export default FileUpload
