"use client"

import { useState } from "react"
import { ChevronDown } from "lucide-react"
import { Modal } from "@/Components/UI/Modal"

const InfoModal = ({ isOpen, onClose, t }) => {
  const [selectedPlatform, setSelectedPlatform] = useState("facebook")
  const [dropdownOpen, setDropdownOpen] = useState(false)

  const platforms = [
    { id: "facebook", name: "Facebook" },
    { id: "instagram", name: "Instagram" },
    { id: "twitter", name: "X (Twitter)" },
    { id: "linkedin", name: "LinkedIn" },
    { id: "youtube", name: "YouTube" },
    { id: "tiktok", name: "TikTok" },
    { id: "threads", name: "Threads" },
  ]

  const disclaimers = {
    facebook: {
      title: "Facebook Posting Guidelines",
      content: [
        "• Posts must comply with Facebook Community Standards",
        "• Avoid misleading or false information",
        "• Respect intellectual property rights",
        "• No spam or repetitive content",
        "• Images should be high quality (1200x630px recommended)",
        "• Videos should be under 4GB and less than 240 minutes",
        "• Use relevant hashtags but avoid overuse",
      ],
      warning: "Facebook may reduce reach for posts that violate community guidelines or appear spammy.",
    },
    instagram: {
      title: "Instagram Posting Guidelines",
      content: [
        "• Follow Instagram Community Guidelines",
        "• Use high-quality images (1080x1080px for feed posts)",
        "• Stories should be 1080x1920px",
        "• Videos should be under 4GB",
        "• Use relevant hashtags (up to 30 per post)",
        "• Avoid shadowbanned hashtags",
        "• Engage authentically with your audience",
      ],
      warning: "Instagram's algorithm favors authentic engagement and may limit reach for inauthentic activity.",
    },
    twitter: {
      title: "X (Twitter) Posting Guidelines",
      content: [
        "• Follow X Rules and Terms of Service",
        "• Character limit: 280 characters per tweet",
        "• Images should be 1200x675px for optimal display",
        "• Videos should be under 512MB",
        "• Use relevant hashtags (2-3 recommended)",
        "• Avoid excessive self-promotion",
        "• Respect others and avoid harassment",
      ],
      warning: "X may limit visibility or suspend accounts that violate platform rules.",
    },
    linkedin: {
      title: "LinkedIn Posting Guidelines",
      content: [
        "• Maintain professional tone and content",
        "• Follow LinkedIn Professional Community Policies",
        "• Images should be 1200x627px for articles",
        "• Videos should be under 5GB",
        "• Focus on industry-relevant content",
        "• Avoid overly promotional content",
        "• Engage meaningfully with professional network",
      ],
      warning: "LinkedIn prioritizes professional, valuable content and may reduce reach for overly promotional posts.",
    },
    youtube: {
      title: "YouTube Posting Guidelines",
      content: [
        "• Follow YouTube Community Guidelines",
        "• Thumbnails should be 1280x720px",
        "• Videos can be up to 12 hours long",
        "• Use relevant tags and descriptions",
        "• Avoid copyright infringement",
        "• Create original, valuable content",
        "• Follow monetization policies if applicable",
      ],
      warning: "YouTube may demonetize or remove content that violates community guidelines.",
    },
    tiktok: {
      title: "TikTok Posting Guidelines",
      content: [
        "• Follow TikTok Community Guidelines",
        "• Videos should be 9:16 aspect ratio",
        "• Maximum video length: 10 minutes",
        "• Use trending sounds and hashtags",
        "• Create engaging, authentic content",
        "• Avoid copyrighted music without permission",
        "• Respect privacy and safety of others",
      ],
      warning: "TikTok's algorithm favors engaging, original content and may limit reach for violations.",
    },
    threads: {
      title: "Threads Posting Guidelines",
      content: [
        "• Follow Threads Community Guidelines",
        "• Character limit: 500 characters per post",
        "• Images should be high quality",
        "• Videos should be under 5 minutes",
        "• Use relevant hashtags sparingly",
        "• Engage authentically with community",
        "• Avoid spam and repetitive content",
      ],
      warning: "Threads prioritizes meaningful conversations and may limit reach for spammy content.",
    },
  }

  const currentDisclaimer = disclaimers[selectedPlatform]

  return (
    <Modal
      isOpen={isOpen}
      onClose={onClose}
      title="Platform Information & Guidelines"
      showSubmit={false}
      size="2xl"
    >
      <div className="space-y-6">
        {/* Platform Selector */}
        <div>
          <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Select Platform</label>
          <div className="relative">
            <button
              onClick={() => setDropdownOpen(!dropdownOpen)}
              className="w-full flex items-center justify-between px-4 py-3 bg-white dark:bg-zinc-800 border border-gray-300 dark:border-gray-600 rounded-lg text-gray-900 dark:text-gray-100 hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
            >
              <span>{platforms.find((p) => p.id === selectedPlatform)?.name}</span>
              <ChevronDown className={`w-4 h-4 transition-transform ${dropdownOpen ? "rotate-180" : ""}`} />
            </button>

            {dropdownOpen && (
              <div className="absolute top-full left-0 right-0 mt-1 bg-white dark:bg-zinc-800 border border-gray-300 dark:border-gray-600 rounded-lg shadow-lg z-10">
                {platforms.map((platform) => (
                  <button
                    key={platform.id}
                    onClick={() => {
                      setSelectedPlatform(platform.id)
                      setDropdownOpen(false)
                    }}
                    className={`w-full text-left px-4 py-3 hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors first:rounded-t-lg last:rounded-b-lg ${
                      selectedPlatform === platform.id
                        ? "bg-cyan-50 dark:bg-cyan-900/20 text-cyan-600 dark:text-cyan-400"
                        : "text-gray-900 dark:text-gray-100"
                    }`}
                  >
                    {platform.name}
                  </button>
                ))}
              </div>
            )}
          </div>
        </div>

        {/* Disclaimer Content */}
        <div className="space-y-4">
          <h3 className="text-lg font-semibold text-gray-900 dark:text-gray-100">{currentDisclaimer.title}</h3>

          <div className="bg-gray-50 dark:bg-zinc-800 rounded-lg p-4">
            <h4 className="font-medium text-gray-900 dark:text-gray-100 mb-3">Guidelines:</h4>
            <ul className="space-y-2 text-sm text-gray-700 dark:text-gray-300">
              {currentDisclaimer.content.map((item, index) => (
                <li key={index} className="leading-relaxed">
                  {item}
                </li>
              ))}
            </ul>
          </div>

          <div className="bg-amber-50 dark:bg-amber-900/20 border border-amber-200 dark:border-amber-800 rounded-lg p-4">
            <h4 className="font-medium text-amber-800 dark:text-amber-200 mb-2 flex items-center">
              <span className="w-2 h-2 bg-amber-500 rounded-full mr-2"></span>
              Important Notice
            </h4>
            <p className="text-sm text-amber-700 dark:text-amber-300 leading-relaxed">{currentDisclaimer.warning}</p>
          </div>

          <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
            <h4 className="font-medium text-blue-800 dark:text-blue-200 mb-2">General Recommendations:</h4>
            <ul className="text-sm text-blue-700 dark:text-blue-300 space-y-1">
              <li>• Always review platform-specific policies before posting</li>
              <li>• Test your content with a small audience first</li>
              <li>• Monitor engagement and adjust strategy accordingly</li>
              <li>• Keep content authentic and valuable to your audience</li>
            </ul>
          </div>
        </div>

        {/* Footer Actions */}
        <div className="flex justify-end pt-4 border-t border-gray-200 dark:border-zinc-800">
          <button
            onClick={onClose}
            className="px-6 py-2 bg-cyan-500 hover:bg-cyan-600 text-white font-medium rounded-lg transition-colors"
          >
            Got it
          </button>
        </div>
      </div>
    </Modal>
  )
}

export default InfoModal
