<?php

namespace App\Models;

use App\Enums\FileKey;
use App\Models\Core\File;
use App\Models\Core\Otp;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
    ];

    protected $appends = ['avatar_url'];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
        ];
    }

    public function currentWorkspace()
    {
        return $this->belongsTo(Workspace::class, 'current_workspace_id');
    }

    /**
     * Get user OTP
     *
     * @return MorphMany
     */
    public function otp(): MorphMany
    {
        return $this->morphMany(Otp::class, 'otpable');
    }

    /**
     * get user files
     *
     * @return MorphOne
     */
    public function files(): MorphOne
    {
        return $this->morphOne(File::class, 'fileable');
    }

    /**
     * get user file (for querying by type)
     *
     * @return MorphMany
     */
    public function file(): MorphMany
    {
        return $this->morphMany(File::class, 'fileable');
    }

    public function filePlatforms()
    {
        return $this->hasMany(UserFilePlatform::class);
    }

    public function getAvatarUrlAttribute()
    {
        $avatar = $this->files()->where('type', FileKey::AVATAR->value)->first();

        if (!$avatar) {
            return null;
        }

        $pathConfig = config('settings.file_path.profile.user.path');

        return asset($pathConfig . '/' . $avatar->name);
    }

    public function role(): BelongsTo
    {
        return $this->belongsTo(Role::class, "role_id", "id")->withDefault([
            'name' => '-',
            "permissions" => json_encode([])
        ]);
    }

    public function workspaces(): HasMany
    {
        return $this->hasMany(Workspace::class);
    }

    public function switchToWorkspace(int $workspaceId)
    {
        $workspace = Workspace::where('id', $workspaceId)
            ->where(function ($query) {
                $query->where('user_id', $this->id)
                    ->orWhere('is_default', true);
            })->first();

        if (!$workspace) {
            throw new \Exception("You do not have access to this workspace.");
        }

        session(['current_workspace_id' => $workspaceId]);

        $this->current_workspace_id = $workspaceId;
        $this->save();

        return true;
    }





}
