<?php

namespace App\Http\Controllers;

use App\Enums\BulkActionEnum;
use App\Enums\StatusEnum;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class RoleController extends Controller
{
    protected $user;

    public function __construct()
    {

        $this->user = auth_user('web');

    }

    public function store(Request $request)
    {
        $request->validate([
            "name" => ["required", "string"]
        ]);


        Role::create([
            'name' => $request->input('name'),
            'created_by' => $this->user->id

        ]);

        return redirect()->back()->with('success', 'User saved');

    }

    public function edit($id)
    {
        // This method is no longer needed as role editing is handled via modal
        // Keeping it for backward compatibility, redirecting to access control page
        return redirect()->route('user.access-control.index');
    }

    public function update(Request $request)
    {

        $request->validate([
            'id' => 'required|integer|exists:roles,id',
            'name' => 'required|string',
            'permissions' => 'nullable|array',
        ]);


        $role = Role::findOrFail($request->input('id'));


        $role->name = $request->input('name');
        $role->permissions = $request->input('permissions');
        $role->save();

        return redirect()->back()->with('success', 'Permissions updated');

    }


    public function delete(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:roles,id'
        ]);

        $role = Role::findOrFail($request->input('id'));
        $role->delete();

        return redirect()->route('user.access-control.index')->with('success', 'Role Deleted');

    }

    public function statusUpdate(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:roles,id',
            'status' => 'required',
        ]);

        try {
            $role = Role::findOrFail($request->id);
            $role->status = $request->status;
            $role->save();

            return back()->with('success', 'Role status updated successfully');
        } catch (\Exception $e) {
            return back()->with('error', 'Role status did not change');

        }
    }

    public function bulk(Request $request)
    {
        $validated = $request->validate([
            'action' => ['required', Rule::in(BulkActionEnum::values())],
            'ids' => 'required|array',
            'ids.*' => 'integer'
        ]);

        $query = Role::whereIn('id', $validated['ids']);

        match (BulkActionEnum::from($validated['action'])) {
            BulkActionEnum::ACTIVATE => $query->update(['status' => StatusEnum::true->status()]),
            BulkActionEnum::INACTIVATE => $query->update(['status' => StatusEnum::false->status()]),
            BulkActionEnum::DELETE => $query->delete(),
        };

        return back()->with('success', "Bulk {$validated['action']} completed.");
    }
}
