<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Core\Language;
use Database\Seeders\Core\LangSeeder;

class TestLangSeeder extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'test:lang-seeder {--reset : Reset languages before testing}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test language seeder and flag display';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Testing Language Seeder...');
        $this->newLine();

        // Reset languages if requested
        if ($this->option('reset')) {
            $this->info('🔄 Resetting languages table...');
            Language::truncate();
            $this->info('✅ Languages table cleared');
            $this->newLine();
        }

        // Show current state
        $this->info('=== Current State ===');
        $currentCount = Language::count();
        $this->info("Current languages in database: {$currentCount}");

        if ($currentCount > 0) {
            $existingLanguages = Language::select('name', 'code', 'lang_code', 'is_default')->get();
            foreach ($existingLanguages as $lang) {
                $default = $lang->is_default === '1' ? ' (DEFAULT)' : '';
                $this->line("  - {$lang->name} | Country Code: {$lang->code} | Lang Code: {$lang->lang_code}{$default}");
            }
        }

        $this->newLine();

        // Run the seeder
        $this->info('🚀 Running LangSeeder...');
        try {
            $seeder = new LangSeeder();
            $seeder->run();
            $this->info('✅ LangSeeder completed');
        } catch (\Exception $e) {
            $this->error('❌ LangSeeder failed: ' . $e->getMessage());
            $this->error('File: ' . $e->getFile() . ' Line: ' . $e->getLine());
            return 1;
        }

        $this->newLine();

        // Show final state
        $this->info('=== Final State ===');
        $finalCount = Language::count();
        $this->info("Final languages in database: {$finalCount}");

        $allLanguages = Language::select('id', 'name', 'code', 'lang_code', 'is_default', 'status', 'ltr')->get();

        foreach ($allLanguages as $lang) {
            $default = $lang->is_default === '1' ? 'DEFAULT' : 'Regular';
            $status = $lang->status === '1' ? 'Active' : 'Inactive';
            $direction = $lang->ltr === '1' ? 'LTR' : 'RTL';

            $this->line("  🌍 {$lang->name}");
            $this->line("     ID: {$lang->id} | Country Code: {$lang->code} | Lang Code: {$lang->lang_code}");
            $this->line("     Status: {$status} | Type: {$default} | Direction: {$direction}");
            $this->newLine();
        }

        // Test flag component compatibility
        $this->info('=== Flag Component Test ===');
        $english = Language::where('lang_code', 'en')->first();

        if ($english) {
            $this->info("✅ English language found:");
            $this->line("   - Name: {$english->name}");
            $this->line("   - Country Code (for flag): {$english->code}");
            $this->line("   - Language Code (for translations): {$english->lang_code}");

            if ($english->code && strlen($english->code) === 2) {
                $this->info("✅ Country code '{$english->code}' is valid for flag display");
            } else {
                $this->error("❌ Country code '{$english->code}' is invalid for flag display");
            }
        } else {
            $this->error("❌ No English language found");
            return 1;
        }

        $this->newLine();
        $this->info('✅ Language seeder test completed!');

        return 0;
    }
}