<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\Workspace;
use App\Enums\StatusEnum;

class TestAdminWorkspace extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'test:admin-workspace';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test admin account and workspace setup after installation';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Testing Admin Account and Workspace Setup...');
        $this->newLine();

        // Test admin user exists
        $this->info('=== Testing Admin User ===');
        try {
            $adminUser = User::where('owner', StatusEnum::true->status())->first();

            if (!$adminUser) {
                $this->error('❌ No admin user found!');
                return 1;
            }

            $this->info("✅ Admin user found: {$adminUser->name} ({$adminUser->email})");
            $this->line("   - ID: {$adminUser->id}");
            $this->line("   - Username: {$adminUser->username}");
            $this->line("   - Owner status: " . ($adminUser->owner === StatusEnum::true->status() ? 'YES' : 'NO'));
            $this->line("   - Current workspace ID: " . ($adminUser->current_workspace_id ?: 'NULL'));
            $this->line("   - Email verified: " . ($adminUser->email_verified_at ? 'YES' : 'NO'));

        } catch (\Exception $e) {
            $this->error('❌ Error checking admin user: ' . $e->getMessage());
            return 1;
        }

        $this->newLine();

        // Test workspace exists
        $this->info('=== Testing Admin Workspace ===');
        try {
            $workspace = Workspace::where('user_id', $adminUser->id)->first();

            if (!$workspace) {
                $this->error('❌ No workspace found for admin user!');
                return 1;
            }

            $this->info("✅ Workspace found: {$workspace->name}");
            $this->line("   - ID: {$workspace->id}");
            $this->line("   - Email: {$workspace->email}");
            $this->line("   - Status: " . ($workspace->status === StatusEnum::true->status() ? 'Active' : 'Inactive'));
            $this->line("   - Is Default: " . ($workspace->is_default === StatusEnum::true->status() ? 'YES' : 'NO'));
            $this->line("   - Is Current: " . ($workspace->is_current === StatusEnum::true->status() ? 'YES' : 'NO'));

        } catch (\Exception $e) {
            $this->error('❌ Error checking workspace: ' . $e->getMessage());
            return 1;
        }

        $this->newLine();

        // Test relationship consistency
        $this->info('=== Testing Relationship Consistency ===');

        if ($adminUser->current_workspace_id === $workspace->id) {
            $this->info('✅ User current_workspace_id matches workspace ID');
        } else {
            $this->error('❌ User current_workspace_id does not match workspace ID');
        }

        // Test workspace relationship
        try {
            $userWorkspace = $adminUser->currentWorkspace;
            if ($userWorkspace && $userWorkspace->id === $workspace->id) {
                $this->info('✅ User->currentWorkspace relationship works correctly');
            } else {
                $this->error('❌ User->currentWorkspace relationship is broken');
            }
        } catch (\Exception $e) {
            $this->error('❌ Error testing currentWorkspace relationship: ' . $e->getMessage());
        }

        // Test workspace user relationship
        try {
            $workspaceUser = $workspace->user;
            if ($workspaceUser && $workspaceUser->id === $adminUser->id) {
                $this->info('✅ Workspace->user relationship works correctly');
            } else {
                $this->error('❌ Workspace->user relationship is broken');
            }
        } catch (\Exception $e) {
            $this->error('❌ Error testing workspace->user relationship: ' . $e->getMessage());
        }

        $this->newLine();

        // Summary
        $this->info('=== Summary ===');
        $totalUsers = User::count();
        $totalWorkspaces = Workspace::count();
        $this->info("Total users in system: {$totalUsers}");
        $this->info("Total workspaces in system: {$totalWorkspaces}");

        $this->newLine();
        $this->info('✅ Admin account and workspace test completed!');

        return 0;
    }
}