import { Heart, MessageCircle, Share, Repeat2 } from "lucide-react"

// Utility functions for hashtag detection and highlighting
const extractHashtags = (text) => {
  const hashtagRegex = /#[a-zA-Z0-9_]+/g
  return text.match(hashtagRegex) || []
}

const highlightContent = (text) => {
  const hashtagRegex = /#[a-zA-Z0-9_]+/g
  const urlRegex = /(https?:\/\/[^\s]+)/g

  // Split by both hashtags and URLs, keeping the delimiters
  const parts = text.split(/(#[a-zA-Z0-9_]+|https?:\/\/[^\s]+)/g)

  return parts
    .map((part, index) => {
      if (!part) return null

      if (hashtagRegex.test(part)) {
        return (
          <span
            key={index}
            className="bg-yellow-200 dark:bg-yellow-600 text-yellow-800 dark:text-yellow-200 px-1 rounded"
          >
            {part}
          </span>
        )
      }

      if (urlRegex.test(part)) {
        return (
          <span key={index} className="text-blue-600 dark:text-blue-400">
            {part}
          </span>
        )
      }

      return part
    })
    .filter(Boolean)
}

export default function TwitterPreview({ postContent = "", postType = "tweet", selectedMedia = [] }) {
  // Mock user data - in real app this would come from selected account
  const mockUser = {
    name: "James Clerk",
    username: "@jamesclerk",
    avatar: "/diverse-user-avatars.png",
    verified: false,
  }

  const hashtags = extractHashtags(postContent)

  return (
    <div className="bg-white dark:bg-zinc-900 rounded-lg border border-gray-200 dark:border-zinc-800 p-4 min-h-[320px] w-full">
      <div className="flex gap-3">
        <img
          src={mockUser.avatar || "/placeholder.svg"}
          alt={mockUser.name}
          className="w-10 h-10 rounded-full flex-shrink-0"
        />
        <div className="flex-1 min-w-0">
          <div className="flex items-center gap-2 mb-1 flex-wrap">
            <span className="font-bold text-gray-900 dark:text-white">{mockUser.name}</span>
            <span className="text-gray-500 dark:text-gray-400 truncate">{mockUser.username}</span>
            <span className="text-gray-500 dark:text-gray-400">·</span>
            <span className="text-gray-500 dark:text-gray-400">now</span>
          </div>

          <div className="text-gray-900 dark:text-white mb-3 whitespace-pre-wrap break-words">
            {postContent ? highlightContent(postContent) : "What's happening?"}
          </div>

          {hashtags.length > 0 && (
            <div className="flex flex-wrap gap-2 mb-3">
              {hashtags.map((hashtag, index) => (
                <span
                  key={index}
                  className="bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200 px-2 py-1 rounded-full text-sm font-medium"
                >
                  {hashtag}
                </span>
              ))}
            </div>
          )}

          {selectedMedia.length > 0 ? (
            <div className="mb-3 rounded-2xl overflow-hidden border border-gray-200 dark:border-gray-600">
              {selectedMedia.length === 1 ? (
                selectedMedia[0].type === "video" ? (
                  <video
                    src={selectedMedia[0].url}
                    poster={selectedMedia[0].thumbnail}
                    controls
                    className="w-full max-h-96 object-cover"
                    preload="metadata"
                  >
                    Your browser does not support the video tag.
                  </video>
                ) : (
                  <img
                    src={selectedMedia[0].thumbnail || selectedMedia[0].url}
                    alt={selectedMedia[0].alt || "Tweet image"}
                    className="w-full max-h-96 object-cover"
                  />
                )
              ) : (
                <div className="grid grid-cols-2 gap-0.5">
                  {selectedMedia.slice(0, 4).map((media, index) => (
                    <div key={media.id} className="relative aspect-video">
                      {media.type === "video" ? (
                        <video
                          src={media.url}
                          poster={media.thumbnail}
                          className="w-full h-full object-cover"
                          preload="metadata"
                          muted
                        >
                          Your browser does not support the video tag.
                        </video>
                      ) : (
                        <img
                          src={media.thumbnail || media.url}
                          alt={media.alt || `Tweet image ${index + 1}`}
                          className="w-full h-full object-cover"
                        />
                      )}
                      {index === 3 && selectedMedia.length > 4 && (
                        <div className="absolute inset-0 bg-black/60 flex items-center justify-center">
                          <span className="text-white text-xl font-bold">+{selectedMedia.length - 4}</span>
                        </div>
                      )}
                    </div>
                  ))}
                </div>
              )}
            </div>
          ) : null}

          <div className="flex items-center justify-between w-full max-w-full">
            <div className="flex items-center gap-1 flex-1">
              <button className="flex items-center gap-1 text-gray-500 dark:text-gray-400 hover:text-blue-500 transition-colors px-2 py-1 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 min-w-0">
                <MessageCircle className="w-4 h-4 flex-shrink-0" />
                <span className="text-sm">0</span>
              </button>
            </div>
            <div className="flex items-center gap-1 flex-1">
              <button className="flex items-center gap-1 text-gray-500 dark:text-gray-400 hover:text-green-500 transition-colors px-2 py-1 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 min-w-0">
                <Repeat2 className="w-4 h-4 flex-shrink-0" />
                <span className="text-sm">0</span>
              </button>
            </div>
            <div className="flex items-center gap-1 flex-1">
              <button className="flex items-center gap-1 text-gray-500 dark:text-gray-400 hover:text-red-500 transition-colors px-2 py-1 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 min-w-0">
                <Heart className="w-4 h-4 flex-shrink-0" />
                <span className="text-sm">0</span>
              </button>
            </div>
            <div className="flex items-center gap-1 flex-1">
              <button className="flex items-center gap-1 text-gray-500 dark:text-gray-400 hover:text-blue-500 transition-colors px-2 py-1 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 min-w-0">
                <Share className="w-4 h-4 flex-shrink-0" />
              </button>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
