import { Heart, MessageCircle, Send, Bookmark, MoreHorizontal } from "lucide-react"

// Utility functions for hashtag detection and highlighting
const extractHashtags = (text) => {
  const hashtagRegex = /#[a-zA-Z0-9_]+/g
  return text.match(hashtagRegex) || []
}

const highlightContent = (text) => {
  const hashtagRegex = /#[a-zA-Z0-9_]+/g
  const urlRegex = /(https?:\/\/[^\s]+)/g

  // Split by both hashtags and URLs, keeping the delimiters
  const parts = text.split(/(#[a-zA-Z0-9_]+|https?:\/\/[^\s]+)/g)

  return parts
    .map((part, index) => {
      if (!part) return null

      if (hashtagRegex.test(part)) {
        return (
          <span
            key={index}
            className="bg-yellow-200 dark:bg-yellow-600 text-yellow-800 dark:text-yellow-200 px-1 rounded"
          >
            {part}
          </span>
        )
      }

      if (urlRegex.test(part)) {
        return (
          <span key={index} className="text-blue-600 dark:text-blue-400">
            {part}
          </span>
        )
      }

      return part
    })
    .filter(Boolean)
}

export default function InstagramPreview({ postContent = "", postType = "feed", selectedMedia = [] }) {
  // Mock user data - in real app this would come from selected account
  const mockUser = {
    name: "James Clerk",
    username: "@jamesclerk",
    avatar: "/diverse-user-avatars.png",
    verified: false,
  }

  const hashtags = extractHashtags(postContent)

  const renderInstagramFeed = () => (
    <div className="bg-white dark:bg-zinc-900 rounded-lg border border-gray-200 dark:border-zinc-800 overflow-hidden min-h-[350px] w-full">
      {/* Header */}
      <div className="p-4 border-b border-gray-100 dark:border-gray-700">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-3">
            <div className="w-8 h-8 rounded-full bg-gradient-to-r from-purple-500 to-pink-500 p-0.5">
              <img
                src={mockUser.avatar || "/placeholder.svg"}
                alt={mockUser.name}
                className="w-full h-full rounded-full border-2 border-white object-cover"
              />
            </div>
            <span className="font-medium text-gray-900 dark:text-white">{mockUser.username}</span>
          </div>
          <MoreHorizontal className="w-5 h-5 text-gray-600 dark:text-gray-400" />
        </div>
      </div>

      {/* Media */}
      {selectedMedia.length > 0 ? (
        <div className="aspect-square">
          {selectedMedia.length === 1 ? (
            selectedMedia[0].type === "video" ? (
              <video
                src={selectedMedia[0].url}
                poster={selectedMedia[0].thumbnail}
                controls
                className="w-full h-full object-cover"
                preload="metadata"
              >
                Your browser does not support the video tag.
              </video>
            ) : (
              <img
                src={selectedMedia[0].thumbnail || selectedMedia[0].url}
                alt={selectedMedia[0].alt || "Post image"}
                className="w-full h-full object-cover"
              />
            )
          ) : (
            <div className="grid grid-cols-2 gap-0.5 h-full">
              {selectedMedia.slice(0, 4).map((media, index) => (
                <div key={media.id} className="relative aspect-square">
                  {media.type === "video" ? (
                    <video
                      src={media.url}
                      poster={media.thumbnail}
                      className="w-full h-full object-cover"
                      preload="metadata"
                      muted
                    >
                      Your browser does not support the video tag.
                    </video>
                  ) : (
                    <img
                      src={media.thumbnail || media.url}
                      alt={media.alt || `Post image ${index + 1}`}
                      className="w-full h-full object-cover"
                    />
                  )}
                  {index === 3 && selectedMedia.length > 4 && (
                    <div className="absolute inset-0 bg-black/60 flex items-center justify-center">
                      <span className="text-white text-xl font-bold">+{selectedMedia.length - 4}</span>
                    </div>
                  )}
                </div>
              ))}
            </div>
          )}
        </div>
      ) : (
        <div className="bg-gray-50 dark:bg-zinc-800 aspect-square flex items-center justify-center text-gray-500 dark:text-gray-400">
          <div className="text-center">
            <div className="w-16 h-16 bg-gradient-to-r from-purple-500 to-pink-500 rounded-full flex items-center justify-center mb-2 mx-auto">
              <div className="w-0 h-0 border-l-[8px] border-l-white border-y-[6px] border-y-transparent ml-1"></div>
            </div>
            <p className="text-sm">No media selected</p>
          </div>
        </div>
      )}

      {/* Actions */}
      <div className="p-4 mt-auto">
        <div className="flex items-center justify-between mb-3">
          <div className="flex items-center gap-4">
            <Heart className="w-6 h-6 text-gray-900 dark:text-white" />
            <MessageCircle className="w-6 h-6 text-gray-900 dark:text-white" />
            <Send className="w-6 h-6 text-gray-900 dark:text-white" />
          </div>
          <Bookmark className="w-6 h-6 text-gray-900 dark:text-white" />
        </div>

        <div className="text-sm text-gray-500 dark:text-gray-400 mb-2">
          <span className="font-medium">0 likes</span>
        </div>

        {/* Caption */}
        <div className="text-sm text-gray-900 dark:text-white break-words">
          <span className="font-medium">{mockUser.username}</span>
          <span className="ml-2">{postContent ? highlightContent(postContent) : "What's in your mind"}</span>
        </div>

        {/* Hashtags display below caption */}
        {hashtags.length > 0 && (
          <div className="flex flex-wrap gap-2 mt-2">
            {hashtags.map((hashtag, index) => (
              <span
                key={index}
                className="bg-purple-100 dark:bg-purple-900 text-purple-800 dark:text-purple-200 px-2 py-1 rounded-full text-sm font-medium"
              >
                {hashtag}
              </span>
            ))}
          </div>
        )}

        <div className="text-xs text-gray-500 dark:text-gray-400 mt-2">
          <span>now</span>
        </div>
      </div>
    </div>
  )

  const renderInstagramReel = () => (
    <div className="bg-black rounded-lg overflow-hidden aspect-[9/16] max-w-[280px] mx-auto relative">
      {selectedMedia.length > 0 ? (
        selectedMedia[0].type === "video" ? (
          <video
            src={selectedMedia[0].url}
            poster={selectedMedia[0].thumbnail}
            className="w-full h-full object-cover"
            autoPlay
            muted
            loop
            playsInline
          >
            Your browser does not support the video tag.
          </video>
        ) : (
          <img
            src={selectedMedia[0].thumbnail || selectedMedia[0].url}
            alt={selectedMedia[0].alt || "Reel image"}
            className="w-full h-full object-cover"
          />
        )
      ) : (
        <div className="absolute inset-0 flex items-center justify-center text-white">
          <div className="text-center">
            <div className="w-16 h-16 bg-gradient-to-r from-purple-500 to-pink-500 rounded-full flex items-center justify-center mb-2 mx-auto">
              <div className="w-0 h-0 border-l-[8px] border-l-white border-y-[6px] border-y-transparent ml-1"></div>
            </div>
            <p className="text-sm">No media selected</p>
          </div>
        </div>
      )}

      {/* Top UI Elements */}
      <div className="absolute top-4 left-4 right-4 flex justify-between items-center">
        <span className="text-white font-medium">Reels</span>
        <div className="w-6 h-6 bg-white/20 rounded-full flex items-center justify-center">
          <MoreHorizontal className="w-4 h-4 text-white" />
        </div>
      </div>

      {/* Bottom UI Elements */}
      <div className="absolute bottom-0 left-0 right-0 p-4 bg-gradient-to-t from-black/80 to-transparent">
        <div className="flex items-end gap-3">
          <div className="flex-1">
            <div className="flex items-center gap-2 mb-2">
              <div className="w-8 h-8 rounded-full bg-gradient-to-r from-purple-500 to-pink-500 p-0.5">
                <img
                  src={mockUser.avatar || "/placeholder.svg"}
                  alt={mockUser.name}
                  className="w-full h-full rounded-full border-2 border-white object-cover"
                />
              </div>
              <span className="text-white font-medium">{mockUser.username}</span>
              <button className="px-3 py-1 border border-white rounded-md text-white text-xs font-medium">
                Follow
              </button>
            </div>
            <p className="text-white text-sm mb-2">
              {postContent ? highlightContent(postContent) : "What's in your mind"}
            </p>
            {hashtags.length > 0 && <div className="text-blue-400 text-sm mb-2">{hashtags.join(" ")}</div>}
            <div className="flex items-center gap-2 text-white text-sm">
              <span>♪ Original audio</span>
            </div>
          </div>

          <div className="flex flex-col gap-4">
            <div className="text-center">
              <button className="w-10 h-10 bg-white/20 rounded-full flex items-center justify-center text-white mb-1">
                <Heart className="w-5 h-5" />
              </button>
              <span className="text-white text-xs">12.5K</span>
            </div>

            <div className="text-center">
              <button className="w-10 h-10 bg-white/20 rounded-full flex items-center justify-center text-white mb-1">
                <MessageCircle className="w-5 h-5" />
              </button>
              <span className="text-white text-xs">234</span>
            </div>

            <div className="text-center">
              <button className="w-10 h-10 bg-white/20 rounded-full flex items-center justify-center text-white mb-1">
                <Send className="w-5 h-5" />
              </button>
            </div>

            <div className="text-center">
              <button className="w-10 h-10 bg-white/20 rounded-full flex items-center justify-center text-white mb-1">
                <Bookmark className="w-5 h-5" />
              </button>
            </div>

            <div className="w-6 h-6 rounded-sm border border-white overflow-hidden">
              <img
                src={selectedMedia[0]?.thumbnail || selectedMedia[0]?.url || "/placeholder.svg"}
                alt="Audio thumbnail"
                className="w-full h-full object-cover"
              />
            </div>
          </div>
        </div>
      </div>
    </div>
  )

  switch (postType) {
    case "reels":
      return renderInstagramReel()
    default:
      return renderInstagramFeed()
  }
}
