"use client"

import { motion } from "framer-motion"
import {
  CheckCircle2,
  Circle,
  Server,
  Database,
  User,
  Rocket
} from "lucide-react"

export default function StepIndicator({ currentStep = 1, completedSteps = [] }) {
  const steps = [
    {
      number: 1,
      title: "Requirements",
      description: "System Check",
      icon: Server,
      path: "/install/requirements"
    },
    {
      number: 2,
      title: "Database",
      description: "Configuration",
      icon: Database,
      path: "/install/db-setup"
    },
    {
      number: 3,
      title: "Account",
      description: "Admin Setup",
      icon: User,
      path: "/install/account/setup"
    },
    {
      number: 4,
      title: "Complete",
      description: "Installation",
      icon: Rocket,
      path: "/install/setup-finished"
    }
  ]

  const getStepStatus = (stepNumber) => {
    if (completedSteps.includes(stepNumber)) return 'completed'
    if (stepNumber === currentStep) return 'active'
    if (stepNumber < currentStep) return 'completed'
    return 'pending'
  }

  return (
    <div className="w-full max-w-4xl mx-auto mb-8">
      <div className="relative">
        {/* Progress Line */}
        <div className="absolute top-8 left-0 right-0 h-0.5 bg-gray-200 dark:bg-gray-700" />

        {/* Active Progress Line */}
        <motion.div
          className="absolute top-8 left-0 h-0.5 bg-gradient-to-r from-blue-500 to-purple-600"
          initial={{ width: '0%' }}
          animate={{
            width: `${((currentStep - 1) / (steps.length - 1)) * 100}%`
          }}
          transition={{ duration: 0.5, ease: "easeInOut" }}
        />

        {/* Steps */}
        <div className="relative flex justify-between">
          {steps.map((step, index) => {
            const status = getStepStatus(step.number)
            const Icon = step.icon
            const isActive = status === 'active'
            const isCompleted = status === 'completed'
            const isPending = status === 'pending'

            return (
              <motion.div
                key={step.number}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1 }}
                className="flex flex-col items-center"
              >
                {/* Step Circle */}
                <div className="relative">
                  <motion.div
                    whileHover={!isPending ? { scale: 1.1 } : {}}
                    className={`
                      w-16 h-16 rounded-full flex items-center justify-center
                      transition-all duration-300 cursor-pointer
                      ${isActive ? 'bg-gradient-to-r from-blue-500 to-purple-600 shadow-lg shadow-blue-500/25' : ''}
                      ${isCompleted ? 'bg-green-500 shadow-lg shadow-green-500/25' : ''}
                      ${isPending ? 'bg-gray-100 dark:bg-gray-800 border-2 border-gray-300 dark:border-gray-600' : ''}
                    `}
                  >
                    {isCompleted ? (
                      <motion.div
                        initial={{ scale: 0, rotate: -180 }}
                        animate={{ scale: 1, rotate: 0 }}
                        transition={{ type: "spring", stiffness: 200 }}
                      >
                        <CheckCircle2 className="w-8 h-8 text-white" />
                      </motion.div>
                    ) : (
                      <Icon className={`
                        w-8 h-8
                        ${isActive ? 'text-white' : ''}
                        ${isPending ? 'text-gray-400 dark:text-gray-500' : ''}
                      `} />
                    )}
                  </motion.div>

                  {/* Pulse Animation for Active Step */}
                  {isActive && (
                    <>
                      <motion.div
                        className="absolute inset-0 rounded-full bg-gradient-to-r from-blue-500 to-purple-600"
                        animate={{
                          scale: [1, 1.2, 1],
                          opacity: [0.5, 0, 0.5]
                        }}
                        transition={{
                          duration: 2,
                          repeat: Infinity,
                          ease: "easeInOut"
                        }}
                      />
                      <motion.div
                        className="absolute inset-0 rounded-full bg-gradient-to-r from-blue-500 to-purple-600"
                        animate={{
                          scale: [1, 1.3, 1],
                          opacity: [0.3, 0, 0.3]
                        }}
                        transition={{
                          duration: 2,
                          repeat: Infinity,
                          ease: "easeInOut",
                          delay: 0.5
                        }}
                      />
                    </>
                  )}
                </div>

                {/* Step Info */}
                <div className="mt-4 text-center">
                  <div className={`
                    text-sm font-semibold
                    ${isActive ? 'text-blue-600 dark:text-blue-400' : ''}
                    ${isCompleted ? 'text-green-600 dark:text-green-400' : ''}
                    ${isPending ? 'text-gray-400 dark:text-gray-500' : ''}
                  `}>
                    Step {step.number}
                  </div>
                  <div className={`
                    text-base font-medium mt-1
                    ${isActive ? 'text-gray-900 dark:text-white' : ''}
                    ${isCompleted ? 'text-gray-700 dark:text-gray-300' : ''}
                    ${isPending ? 'text-gray-400 dark:text-gray-500' : ''}
                  `}>
                    {step.title}
                  </div>
                  <div className={`
                    text-xs mt-0.5
                    ${isActive ? 'text-gray-600 dark:text-gray-400' : ''}
                    ${isCompleted ? 'text-gray-500 dark:text-gray-400' : ''}
                    ${isPending ? 'text-gray-400 dark:text-gray-600' : ''}
                  `}>
                    {step.description}
                  </div>
                </div>

                {/* Connector for Mobile */}
                {index < steps.length - 1 && (
                  <div className="hidden md:block absolute top-8 left-16 w-full">
                    <div className={`
                      h-0.5 transition-all duration-500
                      ${isCompleted ? 'bg-green-500' : ''}
                    `} />
                  </div>
                )}
              </motion.div>
            )
          })}
        </div>
      </div>

      {/* Mobile Step Progress Text */}
      <div className="mt-6 text-center md:hidden">
        <p className="text-sm text-gray-600 dark:text-gray-400">
          Step {currentStep} of {steps.length}
        </p>
      </div>
    </div>
  )
}