"use client"

import { useState } from "react"
import { Button } from "@/Components/UI/Button"
import { Input } from "@/Components/UI/Input"
import { router } from "@inertiajs/react"
import { route } from "ziggy-js"
import useTranslate from "@/hooks/useTranslate";

const LogoSetting = () => {
  const [logoData, setLogoData] = useState({
    site_logo: null,
    user_site_logo: null,
    favicon: null,
    loader_icon: null,
    meta_image: null,
  })

  const t = useTranslate()

  const [isSubmitting, setIsSubmitting] = useState(false)


  const handleFileChange = (field, file) => {
    setLogoData((prev) => ({
      ...prev,
      [field]: file,
    }))
  }

  const handleSubmit = (e) => {
    e.preventDefault()
    setIsSubmitting(true)

    const formData = new FormData()
    Object.entries(logoData).forEach(([key, file]) => {
      if (file) {
        formData.append(`site_settings[${key}]`, file)
      }
    })

    router.post(route('user.setting.store.logo'), formData, {
      preserveState: true,
      preserveScroll: true,
      onFinish: () => setIsSubmitting(false),
      onError: () => setIsSubmitting(false),
    })
  }


  const FileUploadField = ({ label, field, accept = "image/*", description }) => (
    <div className="bg-white dark:bg-zinc-800 rounded-xl border border-gray-200 dark:border-zinc-700 p-6 hover:border-gray-300 dark:hover:border-zinc-600 transition-colors">
      <div className="flex items-center gap-4 mb-4">
        <div className="w-12 h-12 bg-gradient-to-br from-primary/20 to-secondary/20 rounded-lg flex items-center justify-center">
          <svg className="w-6 h-6 text-primary" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
          </svg>
        </div>
        <div>
          <label className="block text-base font-semibold text-gray-900 dark:text-gray-100">{label}</label>
          {description && (
            <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">{description}</p>
          )}
        </div>
      </div>

      <div className="relative">
        <input
          type="file"
          accept={accept}
          onChange={(e) => handleFileChange(field, e.target.files[0])}
          className="absolute inset-0 w-full h-full opacity-0 cursor-pointer z-10"
          id={field}
        />
        <div className="border-2 border-dashed border-gray-300 dark:border-zinc-600 rounded-lg p-6 text-center hover:border-primary dark:hover:border-primary transition-colors">
          {logoData[field] ? (
            <div className="flex items-center justify-center gap-3">
              <svg className="w-8 h-8 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
              <div>
                <p className="text-sm font-medium text-gray-900 dark:text-gray-100">{logoData[field].name}</p>
                <p className="text-xs text-gray-500 dark:text-gray-400">{(logoData[field].size / 1024).toFixed(1)} KB</p>
              </div>
            </div>
          ) : (
            <div>
              <svg className="w-12 h-12 text-gray-400 dark:text-gray-500 mx-auto mb-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12" />
              </svg>
              <p className="text-sm font-medium text-gray-600 dark:text-gray-300 mb-1">{t("Click to upload file")}</p>
              <p className="text-xs text-gray-500 dark:text-gray-400">{t("PNG, JPG, GIF up to 2MB")}</p>
            </div>
          )}
        </div>
      </div>
    </div>
  )

  return (
    <div className="space-y-8">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div>
          <h3 className="text-lg font-semibold text-gray-900 dark:text-gray-100">{t("Brand Management")}</h3>
          <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
            {t("Upload and manage your brand assets for a professional appearance")}
          </p>
        </div>
        <Button
          isLoading={isSubmitting}
          onClick={handleSubmit}
          className="bg-gradient-to-r from-primary to-secondary hover:from-primary/90 hover:to-secondary/90 text-white px-6 py-2.5 rounded-lg font-medium shadow-sm transition-all duration-200 flex items-center gap-2"
        >
          {!isSubmitting && (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
            </svg>
          )}
          {isSubmitting ? t("Submitting...") : t("Save Changes")}
        </Button>
      </div>

      {/* Upload Fields */}
      <div className="grid lg:grid-cols-2 grid-cols-1 gap-6">
        <FileUploadField
          label={t("Site Logo")}
          field="site_logo"
          description={t("Recommended: 150x50px, PNG or SVG format")}
        />
        <FileUploadField
          label={t("Favicon")}
          field="favicon"
          description={t("Recommended: 32x32px, ICO or PNG format")}
        />
        <FileUploadField
          label={t("Loader Icon")}
          field="loader_icon"
          description={t("Recommended: 100x100px, GIF or PNG format")}
        />
        <FileUploadField
          label={t("Meta Image")}
          field="meta_image"
          description={t("Recommended: 1200x630px, for social media sharing")}
        />
      </div>

      {/* Brand Guidelines */}
      <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-xl p-6">
        <div className="flex items-start gap-3">
          <svg className="w-5 h-5 text-blue-600 dark:text-blue-400 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
          </svg>
          <div>
            <h4 className="text-sm font-medium text-blue-900 dark:text-blue-100 mb-2">{t("Brand Guidelines")}</h4>
            <ul className="text-sm text-blue-800 dark:text-blue-200 space-y-1">
              <li>• {t("Use high-quality images with transparent backgrounds when possible")}</li>
              <li>• {t("Ensure logos are readable at different sizes")}</li>
              <li>• {t("Keep file sizes under 2MB for optimal performance")}</li>
              <li>• {t("Use consistent branding across all uploaded assets")}</li>
            </ul>
          </div>
        </div>
      </div>
    </div>
  )
}

export default LogoSetting