import * as React from 'react';
import React__default, { useState, useEffect } from 'react';

/**
 * Hook to load external script.
 * @param src - Source url to load.
 * @param onLoad - Success callback.
 * @param onError - Error callback.
 */ function useLoadScript(src) {
    const [isLoading, setIsLoading] = useState(true);
    const [error, setError] = useState(null);
    const [isSuccess, setIsSuccess] = useState(false);
    const onLoad = ()=>{
        setIsLoading(false);
        setIsSuccess(true);
    };
    useEffect(()=>{
        if (!document) {
            const error = new Error(`[ScriptLoadingError] document not defined when attempting to load ${src}`);
            setError(error);
            return;
        }
        // Find script tag with same src in DOM.
        const foundScript = document.querySelector(`script[src="${src}"]`);
        // Call onLoad if script marked as loaded.
        if (foundScript?.dataset.loaded) {
            onLoad();
            return;
        }
        // Create or get existed tag.
        const script = foundScript || document.createElement("script");
        // Set src if no script was found.
        if (!foundScript) {
            script.src = src;
        }
        // Mark script as loaded on load event.
        const onLoadWithMarker = ()=>{
            script.dataset.loaded = "1";
            onLoad();
        };
        script.addEventListener("load", onLoadWithMarker);
        script.addEventListener("error", (err)=>{
            console.error("Failed to load script:", src, err);
            const error = new Error(`[ScriptLoadingError] Failed to load script: ${src}`);
            setError(error);
        });
        // Add to DOM if not yet added.
        if (!foundScript) {
            document.head.append(script);
        }
    }, []);
    return {
        isLoading,
        error,
        isSuccess
    };
}

const isGoogleReady = (google)=>{
    return google && google.charts;
};
const isGoogleChartsReady = (props, google)=>{
    const { controls, toolbarItems, getChartEditor } = props;
    return google && google.charts && google.visualization && google.visualization.ChartWrapper && google.visualization.Dashboard && (!controls || google.visualization.ChartWrapper) && (!getChartEditor || google.visualization.ChartEditor) && (!toolbarItems || google.visualization.drawToolbar);
};
const getGoogleInstanceFromWindow = (props)=>{
    // @ts-expect-error Getting object from global namespace.
    const google = window.google;
    return google;
};
/**
 * Hook to load Google Charts JS API.
 * @param params - Load parameters.
 * @param [params.chartVersion] - Chart version to load.
 * @param [params.chartPackages] - Packages to load.
 * @param [params.chartLanguage] - Languages to load.
 * @param [params.mapsApiKey] - Google Maps api key.
 * @returns
 */ function useLoadGoogleCharts(props) {
    const { chartVersion = "current", chartPackages = [
        "corechart",
        "controls"
    ], chartLanguage = "en", mapsApiKey } = props;
    const [googleCharts, setGoogleCharts] = useState(null);
    const [scriptInitializationError, setScriptInitializationError] = useState(null);
    const [googleChartsInitializationError, setGoogleChartsInitializationError] = useState(null);
    const { isLoading, error: scriptLoadingError, isSuccess } = useLoadScript(props.chartLoaderScriptUrl || "https://www.gstatic.com/charts/loader.js");
    useEffect(()=>{
        if (!isSuccess) {
            return;
        }
        const google = getGoogleInstanceFromWindow();
        if (!isGoogleReady(google)) {
            const error = new Error("[ScriptInitializationError] Script loaded but Google not attached to window.");
            setScriptInitializationError(error);
            return;
        }
        if (isGoogleChartsReady(props, google)) {
            setGoogleCharts(google);
            return;
        }
        google.charts.load(chartVersion, {
            packages: chartPackages,
            language: chartLanguage,
            mapsApiKey
        });
        google.charts.setOnLoadCallback(()=>{
            if (!isGoogleChartsReady(props, google)) {
                const error = new Error("[GoogleChartsInitializationError] Google Charts not ready after load callback.");
                console.error(error);
                setGoogleChartsInitializationError(error);
                return;
            }
            setGoogleCharts(google);
        });
    }, [
        isSuccess
    ]);
    return {
        error: scriptLoadingError || scriptInitializationError || googleChartsInitializationError,
        isLoading,
        google: googleCharts
    };
}

const chartDefaultProps = {
    // <DEPRECATED_PROPS>
    legend_toggle: false,
    // </DEPRECATED_PROPS>
    options: {},
    legendToggle: false,
    getChartWrapper: ()=>{},
    spreadSheetQueryParameters: {
        headers: 1,
        gid: 1
    },
    rootProps: {},
    chartWrapperParams: {},
    chartLoaderScriptUrl: "https://www.gstatic.com/charts/loader.js"
};

const GoogleChartControls = (props)=>{
    const { isReady, chartControls, filter } = props;
    if (!isReady || !chartControls || !chartControls?.length) {
        return null;
    }
    return /*#__PURE__*/ React__default.createElement(React__default.Fragment, null, chartControls.filter((param)=>{
        let { controlProp, control } = param;
        return filter ? filter({
            control,
            controlProp
        }) : true;
    }).map((param)=>{
        let { control } = param;
        return /*#__PURE__*/ React__default.createElement("div", {
            key: control.getContainerId(),
            id: control.getContainerId()
        });
    }));
};

let uniqueID = 0;
const generateUniqueID = ()=>{
    uniqueID += 1;
    return `reactgooglegraph-${uniqueID}`;
};

/**
 * An internal helper class for creating and managing Google Charts controls.
 * Offers high-level methods to interact with the Google Chart Controls.
 */ class GoogleChartControlsInternal {
    /**
   * Initialize the controls once chart is ready
   */ static initializeControls = (googleChartControls)=>{
        for(let i = 0; i < googleChartControls.length; i += 1){
            const { controlType, options, controlWrapperParams } = googleChartControls[i].controlProp;
            if (controlWrapperParams && "state" in controlWrapperParams) {
                googleChartControls[i].control.setState(controlWrapperParams["state"]);
            }
            googleChartControls[i].control.setOptions(options);
            googleChartControls[i].control.setControlType(controlType);
        }
    };
    /**
   * listen to the control events (ready, statechange, error) specified in the controlEvents prop
   */ static listenToControlEvents = (googleChartControls, props)=>{
        const { google } = props;
        return googleChartControls.flatMap((chartControl)=>{
            const { control, controlProp } = chartControl;
            const { controlEvents = [] } = controlProp;
            return controlEvents.map((event)=>{
                const { callback, eventName } = event;
                return google.visualization.events.addListener(control, eventName, function() {
                    for(var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++){
                        args[_key] = arguments[_key];
                    }
                    callback({
                        chartWrapper: null,
                        controlWrapper: control,
                        props: props,
                        google: google,
                        eventArgs: args
                    });
                });
            });
        });
    };
    /**
   * If controlID is not provided, generate a unique controlID
   */ static createControlId = (id)=>{
        let controlID;
        if (typeof id === "undefined") {
            controlID = `googlechart-control-${generateUniqueID()}`;
        } else {
            controlID = id;
        }
        return controlID;
    };
    /**
   * Map the control props to Google Chart Controls
   */ static createChartControls = (props)=>{
        const { controls, google } = props;
        if (!controls) {
            return null;
        }
        return controls.map((control, i)=>{
            const { controlID: controlIDMaybe, controlType, options: controlOptions, controlWrapperParams } = control;
            const controlID = this.createControlId(controlIDMaybe);
            return {
                controlProp: control,
                control: new google.visualization.ControlWrapper({
                    containerId: controlID,
                    controlType,
                    options: controlOptions,
                    ...controlWrapperParams
                })
            };
        });
    };
    static addControls = (props)=>{
        const { chartWrapper, chartDashboard } = props;
        const googleChartControls = this.createChartControls(props);
        if (!googleChartControls || !chartDashboard || !chartWrapper) {
            return null;
        }
        chartDashboard.bind(googleChartControls.map((param)=>{
            let { control } = param;
            return control;
        }), chartWrapper);
        this.initializeControls(googleChartControls);
        return googleChartControls;
    };
}

const useCreateChartControls = (controls)=>{
    const [chartControls, setChartControls] = React.useState(null);
    const controlAndProp = React.useMemo(()=>{
        if (!chartControls || !controls) return null;
        return controls.map((controlProp, i)=>{
            const control = chartControls[i];
            return control ? {
                controlProp,
                control
            } : undefined;
        }).flatMap((controlAndProp)=>controlAndProp ? [
                controlAndProp
            ] : []);
    }, [
        chartControls,
        controls
    ]);
    return [
        controlAndProp,
        setChartControls
    ];
};
const useListenToControlEvents = (chartControls, props)=>{
    React.useEffect(()=>{
        const listeners = GoogleChartControlsInternal.listenToControlEvents(chartControls ?? [], props);
        return ()=>{
            listeners.forEach((listener)=>{
                props.google.visualization.events.removeListener(listener);
            });
        };
    }, [
        chartControls,
        props
    ]);
};
const useChartControls = (props)=>{
    const [chartControls, setChartControls] = useCreateChartControls(props.controls);
    useListenToControlEvents(chartControls ?? [], props);
    /**
   * Render the container divs for the controls
   */ const renderControl = (filter)=>{
        const { chartWrapper, chartDashboard } = props;
        return /*#__PURE__*/ React.createElement(GoogleChartControls, {
            ...props,
            isReady: Boolean(chartWrapper && chartDashboard),
            chartControls: chartControls,
            filter: filter
        });
    };
    return {
        addControls: (props)=>{
            const controls = GoogleChartControlsInternal.addControls(props);
            setChartControls(controls?.map((control)=>control.control) ?? null);
        },
        renderControl
    };
};

const useChartId = (props)=>{
    const chartIdRef = React.useRef(null);
    const getChartId = ()=>{
        const { graphID, graph_id } = props;
        const chartIdFromProps = graphID || graph_id;
        let currentChartId;
        if (chartIdFromProps) {
            currentChartId = chartIdFromProps;
        } else {
            currentChartId = chartIdRef.current || generateUniqueID();
        }
        chartIdRef.current = currentChartId;
        return chartIdRef.current;
    };
    const chartId = getChartId();
    return {
        chartId
    };
};

const DEFAULT_CHART_COLORS = [
    "#3366CC",
    "#DC3912",
    "#FF9900",
    "#109618",
    "#990099",
    "#3B3EAC",
    "#0099C6",
    "#DD4477",
    "#66AA00",
    "#B82E2E",
    "#316395",
    "#994499",
    "#22AA99",
    "#AAAA11",
    "#6633CC",
    "#E67300",
    "#8B0707",
    "#329262",
    "#5574A6",
    "#3B3EAC"
];

const loadDataTableFromSpreadSheet = async function(googleViz, spreadSheetUrl) {
    let urlParams = arguments.length > 2 && arguments[2] !== void 0 ? arguments[2] : {};
    return new Promise((resolve, reject)=>{
        const headers = `${urlParams.headers ? `headers=${urlParams.headers}` : `headers=0`}`;
        const queryString = `${urlParams.query ? `&tq=${encodeURIComponent(urlParams.query)}` : ``}`;
        const gid = `${urlParams.gid ? `&gid=${urlParams.gid}` : ""}`;
        const sheet = `${urlParams.sheet ? `&sheet=${urlParams.sheet}` : ""}`;
        const access_token = `${urlParams.access_token ? `&access_token=${urlParams.access_token}` : ""}`;
        const urlQueryString = `${headers}${gid}${sheet}${queryString}${access_token}`;
        const urlToSpreadSheet = `${spreadSheetUrl}/gviz/tq?${urlQueryString}`; //&tq=${queryString}`;
        const query = new googleViz.visualization.Query(urlToSpreadSheet);
        query.send((response)=>{
            if (response.isError()) {
                reject(`Error in query:  ${response.getMessage()} ${response.getDetailedMessage()}`);
            } else {
                resolve(response.getDataTable());
            }
        });
    });
};

const GRAY_COLOR = "#CCCCCC";
/**
 * An internal helper class around the Google Chart API.
 * Offers high-level methods to interact with the Google Chart API.
 */ class GoogleChartInternal {
    static grayOutHiddenColumnsLabel = (props, hiddenColumns)=>{
        const { googleChartWrapper, options } = props;
        if (!googleChartWrapper) {
            console.error("googleChartWrapper is not defined");
            return;
        }
        const dataTable = googleChartWrapper.getDataTable();
        if (!dataTable) return;
        const columnCount = dataTable.getNumberOfColumns();
        const hasAHiddenColumn = hiddenColumns.length > 0;
        if (hasAHiddenColumn === false) return;
        const colors = Array.from({
            length: columnCount - 1
        }).map((_dontcare, i)=>{
            const columnID = this.getColumnId(dataTable, i + 1);
            if (hiddenColumns.includes(columnID)) {
                return GRAY_COLOR;
            } else if (options && options.colors) {
                return options.colors[i];
            } else {
                return DEFAULT_CHART_COLORS[i];
            }
        });
        googleChartWrapper.setOptions({
            ...options,
            colors
        });
        googleChartWrapper.draw();
    };
    /**
   * Listens to user clicking on the legend to toggle the visibility of a column.
   * When a user clicks on a legend item, the column id is added to / removed from the hiddenColumns state.
   */ static listenToLegendToggle = (props, hiddenColumnsState)=>{
        const [hiddenColumns, setHiddenColumns] = hiddenColumnsState;
        const { google, googleChartWrapper } = props;
        if (!googleChartWrapper) {
            console.error("googleChartWrapper is not defined");
            return;
        }
        return google.visualization.events.addListener(googleChartWrapper, "select", ()=>{
            const chart = googleChartWrapper.getChart();
            const selection = chart.getSelection();
            const dataTable = googleChartWrapper.getDataTable();
            if (selection.length === 0 || // We want to listen to when a whole row is selected. This is the case only when row === null
            selection[0].row !== null || !dataTable) {
                return;
            }
            const columnIndex = selection[0].column;
            const columnID = this.getColumnId(dataTable, columnIndex);
            // If the column is hidden remove it from state, otherwise add it
            if (hiddenColumns?.includes(columnID)) {
                setHiddenColumns((state)=>[
                        ...state.filter((colID)=>colID !== columnID)
                    ]);
            } else {
                setHiddenColumns((state)=>[
                        ...state,
                        columnID
                    ]);
            }
        });
    };
    /**
   * (Re-)Draw a Google Chart with the given data, options, and chart type.
   */ static draw = async (props)=>{
        const { data, diffdata, rows, columns, options, chartType, formatters, spreadSheetUrl, spreadSheetQueryParameters, googleChartDashboard, googleChartWrapper, google, hiddenColumns, legendToggle, legend_toggle } = props;
        if (!googleChartWrapper) {
            console.error("draw was called with googleChartWrapper = null");
            return;
        }
        let dataTable;
        let chartDiff = null;
        if (diffdata) {
            const oldData = google.visualization.arrayToDataTable(diffdata.old);
            const newData = google.visualization.arrayToDataTable(diffdata.new);
            chartDiff = google.visualization[chartType].prototype.computeDiff(oldData, newData);
        }
        if (data) {
            if (data instanceof google.visualization.DataTable) {
                dataTable = data;
            } else if (Array.isArray(data)) {
                dataTable = google.visualization.arrayToDataTable(data);
            } else {
                dataTable = new google.visualization.DataTable(data);
            }
        } else if (rows && columns) {
            dataTable = google.visualization.arrayToDataTable([
                columns,
                ...rows
            ]);
        } else if (spreadSheetUrl) {
            dataTable = await loadDataTableFromSpreadSheet(google, spreadSheetUrl, spreadSheetQueryParameters);
        } else {
            dataTable = google.visualization.arrayToDataTable([]);
        }
        const columnCount = dataTable.getNumberOfColumns();
        const viewColumns = Array(columnCount).fill(0).map((_c, i)=>{
            const columnID = this.getColumnId(dataTable, i);
            if (hiddenColumns.includes(columnID)) {
                return {
                    label: dataTable.getColumnLabel(i),
                    type: dataTable.getColumnType(i),
                    calc: ()=>null
                };
            } else {
                return i;
            }
        });
        const chart = googleChartWrapper.getChart();
        if (googleChartWrapper.getChartType() === "Timeline") {
            chart && chart.clearChart();
        }
        googleChartWrapper.setChartType(chartType);
        googleChartWrapper.setOptions(options || {});
        const viewTable = new google.visualization.DataView(dataTable);
        viewTable.setColumns(viewColumns);
        googleChartWrapper.setDataTable(viewTable);
        googleChartWrapper.draw();
        if (googleChartDashboard) {
            googleChartDashboard.draw(dataTable);
        }
        if (chartDiff) {
            googleChartWrapper.setDataTable(chartDiff);
            googleChartWrapper.draw();
        }
        if (formatters) {
            this.applyFormatters({
                dataTable,
                formatters,
                google
            });
            googleChartWrapper.setDataTable(dataTable);
            googleChartWrapper.draw();
        }
        if (legendToggle === true || legend_toggle === true) {
            this.grayOutHiddenColumnsLabel(props, hiddenColumns);
        }
        return;
    };
    /**
   * Get the column ID of a column in a GoogleDataTable.
   * If the column has an ID, return the ID, otherwise return the label.
   */ static getColumnId = (dataTable, columnIndex)=>{
        return dataTable.getColumnId(columnIndex) || dataTable.getColumnLabel(columnIndex);
    };
    /**
   * Apply Chart Formatters passed under the formatters prop to the GoogleDataTable
   */ static applyFormatters = (param)=>{
        let { dataTable, formatters, google } = param;
        for (let formatter of formatters){
            switch(formatter.type){
                case "ArrowFormat":
                    {
                        const vizFormatter = new google.visualization.ArrowFormat(formatter.options);
                        vizFormatter.format(dataTable, formatter.column);
                        return;
                    }
                case "BarFormat":
                    {
                        const vizFormatter = new google.visualization.BarFormat(formatter.options);
                        vizFormatter.format(dataTable, formatter.column);
                        return;
                    }
                case "ColorFormat":
                    {
                        const vizFormatter = new google.visualization.ColorFormat(formatter.options);
                        const { ranges } = formatter;
                        if (ranges) {
                            for (let range of ranges){
                                vizFormatter.addRange(...range);
                            }
                        }
                        vizFormatter.format(dataTable, formatter.column);
                        return;
                    }
                case "DateFormat":
                    {
                        const vizFormatter = new google.visualization.DateFormat(formatter.options);
                        vizFormatter.format(dataTable, formatter.column);
                        return;
                    }
                case "NumberFormat":
                    {
                        const vizFormatter = new google.visualization.NumberFormat(formatter.options);
                        vizFormatter.format(dataTable, formatter.column);
                        return;
                    }
                case "PatternFormat":
                    {
                        const vizFormatter = new google.visualization.PatternFormat(formatter.options);
                        vizFormatter.format(dataTable, formatter.column);
                        return;
                    }
                default:
                    {
                        console.warn(`Unknown formatter type: ${formatter.type}`);
                        return;
                    }
            }
        }
    };
}

const useGoogleChartDataTable = (props)=>{
    const { google, googleChartWrapper, googleChartDashboard } = props;
    const [hiddenColumns, setHiddenColumns] = React.useState([]);
    // Re-draw the chart when hiddenColumns change
    React.useEffect(()=>{
        if (!googleChartWrapper) {
            return;
        }
        GoogleChartInternal.draw({
            ...props,
            hiddenColumns,
            googleChartWrapper,
            googleChartDashboard,
            google
        });
    }, [
        hiddenColumns,
        props.data,
        props.rows,
        props.columns,
        props.options,
        props.chartLoaderScriptUrl,
        props.chartType,
        props.formatters,
        props.spreadSheetUrl,
        props.spreadSheetQueryParameters,
        props.legendToggle,
        props.legend_toggle
    ]);
    // Re-draw the chart when the window is resized
    const onResize = ()=>{
        const { googleChartWrapper } = props;
        if (!googleChartWrapper) {
            return;
        }
        googleChartWrapper.draw();
    };
    // Draw the chart when the google charts wrapper is ready and when the hiddenColumns change
    const initialize = (googleChartWrapper)=>{
        const listeners = [];
        const { legendToggle, legend_toggle } = props;
        GoogleChartInternal.draw({
            ...props,
            hiddenColumns,
            googleChartWrapper,
            googleChartDashboard,
            google
        });
        window.addEventListener("resize", onResize);
        if (legend_toggle || legendToggle) {
            const listener = GoogleChartInternal.listenToLegendToggle(props, [
                hiddenColumns,
                setHiddenColumns
            ]);
            if (listener) listeners.push(listener);
        }
        return listeners;
    };
    // Remove event listeners and clear the chart when the component is unmounted
    const destroy = (googleChartWrapper, listeners)=>{
        window.removeEventListener("resize", onResize);
        listeners.forEach((listener)=>{
            google.visualization.events.removeListener(listener);
        });
        if (googleChartWrapper.getChartType() === "Timeline") {
            googleChartWrapper.getChart() && googleChartWrapper.getChart().clearChart();
        }
    };
    React.useEffect(()=>{
        if (!googleChartWrapper) {
            return;
        }
        const listeners = initialize(googleChartWrapper);
        return ()=>{
            destroy(googleChartWrapper, listeners);
        };
    }, [
        googleChartWrapper,
        initialize,
        destroy
    ]);
};

const listenToEvents = (props)=>{
    const { chartEvents, google, googleChartWrapper } = props;
    if (!chartEvents) {
        return;
    }
    if (!googleChartWrapper) {
        console.warn("listenToEvents was called before chart wrapper ready.");
        return;
    }
    return chartEvents.map((param)=>{
        let { eventName, callback } = param;
        return google.visualization.events.addListener(googleChartWrapper, eventName, function() {
            for(var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++){
                args[_key] = arguments[_key];
            }
            callback({
                chartWrapper: googleChartWrapper,
                props,
                google: google,
                eventArgs: args
            });
        });
    });
};
const useGoogleChartEvents = (props)=>{
    useEffect(()=>{
        if (!props.googleChartWrapper) return;
        const listeners = listenToEvents(props);
        return ()=>{
            listeners?.forEach((listener)=>{
                props.google.visualization.events.removeListener(listener);
            });
        };
    }, [
        props
    ]);
};

const GoogleChart = (props)=>{
    const [googleChartWrapper, setGoogleChartWrapper] = React.useState(null);
    // const [isReady, setIsReady] = React.useState<boolean>(false);
    const [googleChartDashboard, setGoogleChartDashboard] = React.useState(null);
    const { addControls, renderControl } = useChartControls({
        ...props,
        chartDashboard: googleChartDashboard,
        chartWrapper: googleChartWrapper
    });
    useGoogleChartEvents({
        ...props,
        googleChartWrapper
    });
    const { chartId } = useChartId(props);
    const dashboardRef = React.useRef(null);
    const toolbarRef = React.useRef(null);
    React.useEffect(()=>{
        const { options, google, chartType, chartWrapperParams, toolbarItems, getChartEditor, getChartWrapper, onLoad } = props;
        const chartConfig = {
            chartType,
            options,
            containerId: chartId,
            ...chartWrapperParams
        };
        // Create ChartWrapper instance, pass it to the user and store it in state
        const chartWrapper = new google.visualization.ChartWrapper(chartConfig);
        chartWrapper.setOptions(options || {});
        getChartWrapper?.(chartWrapper, google);
        // Create Dashboard instance, needed for controls
        const chartDashboard = new google.visualization.Dashboard(dashboardRef.current);
        // Create toolbar if needed
        if (toolbarItems) {
            google.visualization.drawToolbar(toolbarRef.current, toolbarItems);
        }
        // Create ChartEditor instance if needed and pass it to the user
        let chartEditor = null;
        if (getChartEditor) {
            chartEditor = new google.visualization.ChartEditor();
            getChartEditor({
                chartEditor,
                chartWrapper,
                google
            });
        }
        // Create and add controls to the chart / dashboard
        addControls({
            ...props,
            chartDashboard,
            chartWrapper
        });
        setGoogleChartWrapper(chartWrapper);
        setGoogleChartDashboard(chartDashboard);
        onLoad?.(google, {
            google,
            chartWrapper,
            chartEditor,
            chartDashboard
        });
    }, []);
    useGoogleChartDataTable({
        ...props,
        googleChartWrapper,
        googleChartDashboard
    });
    const renderChart = ()=>{
        const { width, height, options, style, className, rootProps, google } = props;
        const divStyle = {
            height: height || options && options.height,
            width: width || options && options.width,
            ...style
        };
        return /*#__PURE__*/ React.createElement("div", {
            id: chartId,
            style: divStyle,
            className: className,
            ...rootProps
        });
    };
    const renderToolBar = ()=>{
        if (!props.toolbarItems) return null;
        return /*#__PURE__*/ React.createElement("div", {
            ref: toolbarRef
        });
    };
    const { width, height, options, style } = props;
    const divStyle = {
        height: height || options && options.height,
        width: width || options && options.width,
        ...style
    };
    // If render prop is provided, give the user full control over the rendering by passing renderChart, renderControl and renderToolbar functions
    if (props.render) {
        return /*#__PURE__*/ React.createElement("div", {
            ref: dashboardRef,
            style: divStyle
        }, /*#__PURE__*/ React.createElement("div", {
            ref: toolbarRef,
            id: "toolbar"
        }), props.render({
            renderChart,
            renderControl,
            renderToolbar: renderToolBar
        }));
    } else {
        return /*#__PURE__*/ React.createElement("div", {
            ref: dashboardRef,
            style: divStyle
        }, renderControl((param)=>{
            let { controlProp } = param;
            return controlProp.controlPosition !== "bottom";
        }), renderChart(), renderControl((param)=>{
            let { controlProp } = param;
            return controlProp.controlPosition === "bottom";
        }), renderToolBar());
    }
};

const ChartContext = /*#__PURE__*/ React.createContext(chartDefaultProps);
const ContextProvider = (param)=>{
    let { children, value } = param;
    return /*#__PURE__*/ React.createElement(ChartContext.Provider, {
        value: value
    }, children);
};

/**
 * Loads Google Charts JS and renders the GoogleChart component.
 */ const ChartView = (props)=>{
    const { google, isLoading, error } = useLoadGoogleCharts(props);
    if (isLoading) {
        return props.loader ?? null;
    }
    if (error) {
        return props.errorElement ?? null;
    }
    if (google) {
        return /*#__PURE__*/ React__default.createElement(GoogleChart, {
            google: google,
            ...props
        });
    }
    return null;
};
/**
 * Updates the context with the props and renders ChartView.
 */ const Chart = (userProps)=>{
    const props = {
        ...chartDefaultProps,
        ...userProps
    };
    return /*#__PURE__*/ React__default.createElement(ContextProvider, {
        value: props
    }, /*#__PURE__*/ React__default.createElement(ChartView, props));
};

// Complete Google Charts Type Definition : https://github.com/DefinitelyTyped/DefinitelyTyped/blob/master/types/google.visualization/index.d.ts
var GoogleDataTableColumnRoleType;
(function(GoogleDataTableColumnRoleType) {
    GoogleDataTableColumnRoleType["annotation"] = "annotation";
    GoogleDataTableColumnRoleType["annotationText"] = "annotationText";
    GoogleDataTableColumnRoleType["certainty"] = "certainty";
    GoogleDataTableColumnRoleType["emphasis"] = "emphasis";
    GoogleDataTableColumnRoleType["interval"] = "interval";
    GoogleDataTableColumnRoleType["scope"] = "scope";
    GoogleDataTableColumnRoleType["style"] = "style";
    GoogleDataTableColumnRoleType["tooltip"] = "tooltip";
    GoogleDataTableColumnRoleType["domain"] = "domain";
})(GoogleDataTableColumnRoleType || (GoogleDataTableColumnRoleType = {}));

export { Chart, GoogleDataTableColumnRoleType, Chart as default };
//# sourceMappingURL=index.js.map
