<?php

namespace App\Http\Controllers\Auth;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Services\Auth\AuthService;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use Inertia\Inertia;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;

class NewPasswordController extends Controller
{

    private $authService;
    public function __construct()
    {
        $this->authService = new AuthService();
    }


    /**
     * forget password
     *
     * 
     */
    public function create()
    {

        return Inertia::render('Auth/ForgotPassword');
    }


    /**
     * forget password
     *
     * @return mixed
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'email' => "required|email|exists:users,email"
        ]);


        $user = User::where('email', $request->input('email'))->firstOrfail();
        $response = $this->authService->sendOtp($user);

        session()->put("user_reset_password_email", $user->email);
        return redirect()->route("auth.password.verify");

    }



    /**
     * return verification route
     *
     * @return
     */
    public function verify()
    {

        return Inertia::render('Auth/VerifyCode');
    }


    /**
     * @param Request $request
     * @return mixed
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function verifyCode(Request $request): mixed
    {
        $request->validate([
            'otp_code' => "required|numeric|exists:otps,otp",
        ]);

        $response = response_status("The provided code does not exist.", 'error');

        $user = User::with("otp")
            ->where('email', session()->get("user_reset_password_email"))
            ->firstOrfail();

        $otp = $user->otp->where("otp", $request->input("otp_code"))->first();

        if ($otp) {
            session()->put("user_reset_password_code", $request->input("otp_code"));
            return redirect()->route('auth.password.reset');
        }

        return redirect()->back()->with($response);
    }

    /**
     * reset view
     *
     * @return
     */
    public function resetPassword()
    {

        return Inertia::render('Auth/ResetPassword');
        
    }


    /**
     * @param Request $request
     * @return RedirectResponse
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function updatePassword(Request $request): RedirectResponse
    {

        $rules['password'] = ["required", "confirmed", Password::min(3)];

        if (site_settings('strong_password') == StatusEnum::true->status()) {

            $rules['password'] = [
                "required",
                "confirmed",
                Password::min(6)
                    ->mixedCase()
                    ->letters()
                    ->numbers()
                    ->symbols()
                    ->uncompromised()
            ];
        }

        $request->validate($rules);
        $response = response_status("Invalid otp !! please verify your otp again!!", 'error');
        $user = User::with("otp")
            ->where('email', session()->get("user_reset_password_email"))
            ->firstOrfail();

        $otp = $user->otp->where("otp", session()->get("user_reset_password_code"))->first();
        if ($otp) {
            $response = response_status("Your Password Has Been Updated!!");
            $user->password = $request->input('password');
            $user->save();
            $otp->delete();
            session()->forget("user_reset_password_code");
            session()->forget("user_reset_password_email");
        }

        return redirect()->route('auth.password.update.success')->with($response);

    }

    public function updatePasswordSuccess()
    {
        return Inertia::render('Auth/PasswordResetSuccess');
    }

}
